/* This file is part of the KDE project
   Copyright (C) 2004-2006 David Faure <faure@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License version 2 as published by the Free Software Foundation.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#ifndef KOOASISSTYLES_H
#define KOOASISSTYLES_H

#include <tqdom.h>
#include <tqdict.h>
#include <tqvaluevector.h>
#include <tqmap.h>
#include <koffice_export.h>

class KoGenStyles;
class KoXmlWriter;
class TQBrush;
class KoGenStyle;
class KoStyleStack;

/**
 * Repository of styles used during loading of OASIS/OOo file
 */
class KOFFICECORE_EXPORT KoOasisStyles
{
public:
    KoOasisStyles();
    ~KoOasisStyles();

    /// Look into @p doc for styles and remember them
    /// @param doc document to look into
    /// @param stylesDotXml true when loading styles.xml, false otherwise
    void createStyleMap( const TQDomDocument& doc, bool stylesDotXml );

    /**
     * Look up a style by name.
     *  This method can find styles defined by the tags "style:page-layout",
     *   "style:presentation-page-layout", or "style:font-decl".
     * Do NOT use this method for style:style styles.
     *
     * @param name the style name
     * @return the dom element representing the style, or TQString() if it wasn't found.
     */
    const TQDomElement* findStyle( const TQString& name ) const;

    /**
     * Look up a style:style by name.
     * @param name the style name
     * @param family the style family (for a style:style, use 0 otherwise)
     * @return the dom element representing the style, or TQString() if it wasn't found.
     */
    const TQDomElement* findStyle( const TQString& name, const TQString& family ) const;

    /// Similar to findStyle but for auto-styles in styles.xml only.
    const TQDomElement* findStyleAutoStyle( const TQString& name, const TQString& family ) const;

    /// @return the style:styles that are "user styles", i.e. those from office:styles
    /// findStyle() is used for lookup. userStyles() is used to load all user styles upfront.
    TQValueVector<TQDomElement> userStyles() const;

    /// @return the default style for a given family ("graphic","paragraph","table" etc.)
    /// Returns 0 if no default style for this family is available
    const TQDomElement* defaultStyle( const TQString& family ) const;

    /// @return the office:style element
    const TQDomElement& officeStyle() const { return m_officeStyle; }

    /// @return all list styles ("text:list-style" elements), hashed by name
    const TQDict<TQDomElement>& listStyles() const { return m_listStyles; }

    /// @return master pages ("style:master-page" elements), hashed by name
    const TQDict<TQDomElement>& masterPages() const { return m_masterPages; }

    /// @return draw styles, hashed by name
    const TQDict<TQDomElement>& drawStyles() const { return m_drawStyles; }

    /// @return all styles ("style:style" elements) for a given family, hashed by name
    const TQDict<TQDomElement>& styles(const TQString& family) const;

    /// Prefix and suffix are always included into formatStr. Having them as separate fields simply
    /// allows to extract them from formatStr, to display them in separate widgets.
    struct NumericStyleFormat
    {
        TQString formatStr;
        TQString prefix;
        TQString suffix;
        enum { Number, Scientific, Fraction, Currency, Percentage,
               Date, Time, Boolean, Text } type;
        int precision;
        TQString currencySymbol;
    };

    typedef TQMap<TQString, NumericStyleFormat> DataFormatsMap;
    /// Value (date/time/number...) formats found while parsing styles. Used e.g. for fields.
    /// Key: format name. Value:
    const DataFormatsMap& dataFormats() const { return m_dataFormats; }

    static TQString saveOasisDateStyle( KoGenStyles &mainStyles, const TQString & _format, bool klocaleFormat,
                                       const TQString &_prefix = TQString() , const TQString &_suffix= TQString() );
    static TQString saveOasisTimeStyle( KoGenStyles &mainStyles, const TQString & _format, bool klocaleFormat,
                                       const TQString &_prefix = TQString() , const TQString &_suffix= TQString() );
    static TQString saveOasisFractionStyle( KoGenStyles &mainStyles, const TQString & _format,
                                           const TQString &_prefix = TQString() , const TQString &_suffix= TQString() );
    static TQString saveOasisScientificStyle( KoGenStyles &mainStyles, const TQString & _format,
                                             const TQString &_prefix = TQString() , const TQString &_suffix= TQString() );
    static TQString saveOasisNumberStyle( KoGenStyles &mainStyles, const TQString & _format,
                                         const TQString &_prefix = TQString() , const TQString &_suffix= TQString() );
    static TQString saveOasisPercentageStyle( KoGenStyles &mainStyles, const TQString & _format,
                                             const TQString &_prefix = TQString() , const TQString &_suffix= TQString() );
    static TQString saveOasisCurrencyStyle( KoGenStyles &mainStyles, const TQString & _format, const TQString &symbol,
                                           const TQString &_prefix = TQString() , const TQString &_suffix= TQString() );
    static TQString saveOasisTextStyle( KoGenStyles &mainStyles, const TQString & _format,
                                       const TQString &_prefix = TQString() , const TQString &_suffix= TQString() );

    static void saveOasisFillStyle( KoGenStyle &styleFill, KoGenStyles& mainStyles, const TQBrush & brush );
    static TQString saveOasisHatchStyle( KoGenStyles& mainStyles, const TQBrush &brush );

    static TQBrush loadOasisFillStyle( const KoStyleStack &styleStack, const TQString & fill,  const KoOasisStyles & oasisStyles );

private:
    /// Add styles to styles map
    void insertStyles( const TQDomElement& styles, bool styleAutoStyles = false );

private:
    void insertOfficeStyles( const TQDomElement& styles );
    void insertStyle( const TQDomElement& style, bool styleAutoStyles );
    void importDataStyle( const TQDomElement& parent );
    static bool saveOasisTimeFormat( KoXmlWriter &elementWriter, TQString & format, TQString & text, bool &antislash );
    static void parseOasisDateKlocale(KoXmlWriter &elementWriter, TQString & format, TQString & text );
    static bool saveOasisKlocaleTimeFormat( KoXmlWriter &elementWriter, TQString & format, TQString & text );
    static void parseOasisTimeKlocale(KoXmlWriter &elementWriter, TQString & format, TQString & text );
    static void addKofficeNumericStyleExtension( KoXmlWriter & elementWriter, const TQString &_suffix, const TQString &_prefix );

    KoOasisStyles( const KoOasisStyles & ); // forbidden
    KoOasisStyles& operator=( const KoOasisStyles & ); // forbidden

private:
    TQDict<TQDomElement>   m_styles; // page-layout, font-decl etc.
    TQDict<TQDomElement>   m_defaultStyle;
    TQDomElement m_officeStyle;

    TQDict<TQDomElement>   m_masterPages;
    TQDict<TQDomElement>   m_listStyles;

    TQDict<TQDomElement>   m_drawStyles;
    DataFormatsMap m_dataFormats;

    class Private;
    Private * const d;
};

#endif /* KOOASISSTYLES_H */
