/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 *
 * Date        : 2005-07-23
 * Description : simple plugins dialog without threadable
 *               filter interface. The dialog layout is
 *               designed to accept custom widgets in
 *               preview and settings area.
 *
 * Copyright (C) 2005-2007 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

// TQt includes.

#include <tqgroupbox.h>
#include <tqlabel.h>
#include <tqwhatsthis.h>
#include <tqpushbutton.h>
#include <tqtimer.h>
#include <tqlayout.h>
#include <tqframe.h>
#include <tqtimer.h>
#include <tqsplitter.h>
#include <tqhbox.h>

// KDE includes.

#include <kcursor.h>
#include <tdelocale.h>
#include <tdeglobalsettings.h>
#include <tdeaboutdata.h>
#include <khelpmenu.h>
#include <kiconloader.h>
#include <tdeapplication.h>
#include <tdepopupmenu.h>
#include <tdestandarddirs.h>
#include <tdeconfig.h>

// Local includes.

#include "ddebug.h"
#include "sidebar.h"
#include "dimginterface.h"
#include "imagedlgbase.h"
#include "imagedlgbase.moc"

namespace Digikam
{

class ImageDlgBasePriv
{
public:

    ImageDlgBasePriv()
    {
        aboutData       = 0;
        timer           = 0;
        parent          = 0;
        mainLayout      = 0;
        hbox            = 0;
        settingsSideBar = 0;
        splitter        = 0;
    }

    bool            tryAction;

    TQGridLayout    *mainLayout;
    
    TQWidget        *parent;
    
    TQString         name;

    TQTimer         *timer;

    TQHBox          *hbox;

    TQSplitter      *splitter;

    TDEAboutData     *aboutData;

    Sidebar        *settingsSideBar;
};

ImageDlgBase::ImageDlgBase(TQWidget* parent, TQString title, TQString name, 
                           bool loadFileSettings, bool tryAction, TQFrame* bannerFrame)
            : KDialogBase(Plain, 0, Help|Default|User1|User2|User3|Try|Ok|Cancel, Ok,
                          parent, 0, true, true,
                          TQString(),
                          i18n("&Save As..."),
                          i18n("&Load..."))
{
    tdeApp->setOverrideCursor( KCursor::waitCursor() );
    setCaption(DImgInterface::defaultInterface()->getImageFileName() + TQString(" - ") + title);
    showButton(User1, false);

    d = new ImageDlgBasePriv;
    d->parent    = parent;
    d->name      = name;
    d->tryAction = tryAction;

    setButtonWhatsThis ( Default, i18n("<p>Reset all filter parameters to their default values.") );
    setButtonWhatsThis ( User3, i18n("<p>Load all filter parameters from settings text file.") );
    setButtonWhatsThis ( User2, i18n("<p>Save all filter parameters to settings text file.") );
    showButton(User2, loadFileSettings);
    showButton(User3, loadFileSettings);
    showButton(Try, tryAction);

    resize(configDialogSize(name + TQString(" Tool Dialog")));

    // -------------------------------------------------------------

    d->mainLayout = new TQGridLayout( plainPage(), 2, 1);
    if (bannerFrame)
    {
        bannerFrame->reparent( plainPage(), TQPoint(0, 0) );
        d->mainLayout->addMultiCellWidget(bannerFrame, 0, 0, 0, 1);
    }

    // -------------------------------------------------------------

    d->hbox     = new TQHBox(plainPage());
    d->splitter = new TQSplitter(d->hbox);
    d->splitter->setFrameStyle( TQFrame::NoFrame );
    d->splitter->setFrameShadow( TQFrame::Plain );
    d->splitter->setFrameShape( TQFrame::NoFrame );    
    d->splitter->setOpaqueResize(false);
    
    d->mainLayout->addMultiCellWidget(d->hbox, 1, 2, 0, 1);
    d->mainLayout->setColStretch(0, 10);
    d->mainLayout->setRowStretch(2, 10);

    tdeApp->restoreOverrideCursor();
}

ImageDlgBase::~ImageDlgBase()
{
    if (d->timer)
       delete d->timer;

    if (d->aboutData)
       delete d->aboutData;

    delete d->settingsSideBar;
    delete d;
}

void ImageDlgBase::readSettings(void)
{
    TDEConfig *config = tdeApp->config();
    config->setGroup(d->name + TQString(" Tool Dialog"));
    if(config->hasKey("SplitterSizes"))
        d->splitter->setSizes(config->readIntListEntry("SplitterSizes"));

    readUserSettings();
}

void ImageDlgBase::writeSettings()
{
    TDEConfig *config = tdeApp->config();
    config->setGroup(d->name + TQString(" Tool Dialog"));
    config->writeEntry("SplitterSizes", d->splitter->sizes());
    config->sync();
    saveDialogSize(d->name + TQString(" Tool Dialog"));
}

void ImageDlgBase::closeEvent(TQCloseEvent *e)
{
    writeSettings();
    e->accept();
}

void ImageDlgBase::slotCancel()
{
    writeSettings();
    done(Cancel);
}

void ImageDlgBase::slotOk()
{
    writeSettings();
    writeUserSettings();
    finalRendering();
}

void ImageDlgBase::slotDefault()
{
    resetValues();
    slotEffect();
}

void ImageDlgBase::slotHelp()
{
    // If setAboutData() is called by plugin, well DigikamImagePlugins help is launched, 
    // else digiKam help. In this case, setHelp() method must be used to set anchor and handbook name.

    if (d->aboutData)
        tdeApp->invokeHelp(d->name, "digikam");
    else
        KDialogBase::slotHelp();
}

void ImageDlgBase::setAboutData(TDEAboutData *about)
{
    d->aboutData = about;
    TQPushButton *helpButton = actionButton( Help );
    KHelpMenu* helpMenu = new KHelpMenu(this, d->aboutData, false);
    helpMenu->menu()->removeItemAt(0);
    helpMenu->menu()->insertItem(i18n("digiKam Handbook"), this, TQ_SLOT(slotHelp()), 0, -1, 0);
    helpButton->setPopup( helpMenu->menu() );
}

void ImageDlgBase::setPreviewAreaWidget(TQWidget *w)
{
    w->reparent( d->splitter, TQPoint(0, 0) );
    TQSizePolicy rightSzPolicy(TQSizePolicy::Preferred,
                              TQSizePolicy::Expanding,
                              2, 1);
    w->setSizePolicy(rightSzPolicy);
}

void ImageDlgBase::setUserAreaWidget(TQWidget *w)
{
    TQString sbName(d->name + TQString(" Image Plugin Sidebar"));
    d->settingsSideBar = new Sidebar(d->hbox, sbName.ascii(), Sidebar::Right);
    d->settingsSideBar->setSplitter(d->splitter);
    d->settingsSideBar->appendTab(w, SmallIcon("configure"), i18n("Settings"));    
    d->settingsSideBar->loadViewState();

    readSettings();   
}

void ImageDlgBase::slotTimer()
{
    if (d->timer)
    {
       d->timer->stop();
       delete d->timer;
    }

    d->timer = new TQTimer( this );
    connect( d->timer, TQ_SIGNAL(timeout()),
             this, TQ_SLOT(slotEffect()) );
    d->timer->start(500, true);
}

}  // NameSpace Digikam

