/*
 *   kiosksync.cpp
 *
 *   Copyright (C) 2004 Waldo Bastian <bastian@kde.org>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License version 2 as
 *   published by the Free Software Foundation.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "kiosksync.h"

#include <tqdir.h>
#include <tqwidget.h>

#include <kdebug.h>
#include <tdelocale.h>
#include <tdesimpleconfig.h>
#include <tdestandarddirs.h>

KioskSync::KioskSync( TQWidget* parent, const char* name)
 : TQObject(parent, name),
   m_parent(parent)
{
}

KioskSync::~KioskSync()
{
}

void
KioskSync::addDir(const TQString &_src, const KURL &dest)
{
   TQString src = _src;
   if (!src.endsWith("/"))
      src.append("/");
   
   m_syncDirs.append(SyncDir(src, dest));
}

bool
KioskSync::sync(bool incremental)
{
   m_incremental = incremental;
   m_timestamps = new TDESimpleConfig(locateLocal("appdata", "profile-data"));

   bool canceled = false; 
 
   for(SyncDirList::ConstIterator it = m_syncDirs.begin();
       it != m_syncDirs.end(); ++it)
   {
      m_changedFiles.clear();
      m_changedDirs.clear();
      
      m_timestamps->setGroup((*it).src);

      if (!KioskRun::self()->createRemoteDirRecursive((*it).dest, true))
      {
         canceled = true;
         break;
      }
      
      scanChangedFiles((*it).src, TQString());

      for(TQStringList::ConstIterator it2 = m_changedDirs.begin();
          it2 != m_changedDirs.end(); ++it2)
      {
         KURL dest = (*it).dest;
         dest.setPath(dest.path(1) + *it2);
         if (!KioskRun::self()->createRemoteDir(dest))
         {
            canceled = true;
            break;
         }
      }

      if (canceled)
         break;

      for(TQStringList::ConstIterator it2 = m_changedFiles.begin();
          it2 != m_changedFiles.end(); ++it2)
      {
         KURL dest = (*it).dest;
         dest.setPath(dest.path(1) + *it2);
         if (!syncFile((*it).src, *it2, dest))
         {
            canceled = true;
            break;
         }
      }
      if (canceled)
         break;
   }
   delete m_timestamps;
   m_changedFiles.clear();
   m_changedDirs.clear();
   
   return !canceled;
}

TQStringList
KioskSync::listFiles()
{
   m_changedFiles.clear();
   m_changedDirs.clear();
   m_incremental = false;
   m_timestamps = 0;

   for(SyncDirList::ConstIterator it = m_syncDirs.begin();
       it != m_syncDirs.end(); ++it)
   {
      scanChangedFiles((*it).src, TQString());
   }
   return m_changedFiles;
}

void
KioskSync::addChangedDir(const TQString &dir)
{
   if (dir.isEmpty())
      return;

   if (m_changedDirs.contains(dir))
      return;

   int i = dir.findRev('/', -2);
   if (i != -1)
   {
       TQString parentDir = dir.left(i+1);
       addChangedDir(parentDir);
   }

   kdDebug() << "KioskSync: Adding " << dir << endl;
   m_changedDirs.append(dir);
}

void
KioskSync::scanChangedFiles(const TQString &_dir, const TQString &prefix)
{
   kdDebug() << "KioskSync: Scanning " << _dir << endl;
   TQDir dir(_dir);
   if (!dir.exists())
   {
       emit warning(i18n("Directory <b>%1</b> does not exist.").arg(_dir));
       return;
   }
   if (!dir.isReadable())
   {
       emit warning(i18n("Directory <b>%1</b> is not readable.").arg(_dir));
       return;
   }

   TQStringList subDirs;
   const TQFileInfoList *list = dir.entryInfoList(TQDir::Dirs | TQDir::Files | TQDir::NoSymLinks);

   bool dirtyDir = false;   
   TQFileInfoListIterator it( *list );
   for ( TQFileInfo *fi; (fi = it.current()) != 0; ++it)
   {
       if (fi->isDir())
       {
          TQString subDir = fi->fileName();
          if ((subDir != ".") && (subDir != ".."))
             subDirs.append(subDir+"/");
          continue;
       }
 
       // TODO: Check file
       TQString file = prefix + fi->fileName();
       TQDateTime lastModified = fi->lastModified();
       if (!m_incremental || !m_timestamps->hasKey(file) || 
           (m_timestamps->readDateTimeEntry(file) != lastModified))
       {
          dirtyDir = true;
          m_changedFiles.append(file);
       }
   }
   if (dirtyDir)
      addChangedDir(prefix);
   
   for( TQStringList::ConstIterator it = subDirs.begin();
        it != subDirs.end(); ++it)
   {
       TQString subDir = *it;
       scanChangedFiles(_dir + subDir, prefix + subDir);
   }
}

bool
KioskSync::syncFile(const TQString &prefix, const TQString &file, const KURL &dest)
{
   kdDebug() << "KioskSync: Syncing [" << prefix << "]" << file << " --> " << dest.prettyURL() << endl;
   
   if (!KioskRun::self()->uploadRemote(prefix+file, dest))
       return false;
   
   TQFileInfo fi(prefix+file);
   m_timestamps->writeEntry(file, fi.lastModified());
   return true;
}

#include "kiosksync.moc"
