/* This file is part of the KDE project
   Copyright (C) 1998, 1999 Reginald Stadlbauer <reggie@kde.org>
   Copyright 2001, 2002 Nicolas GOUTTE <goutte@kde.org>
   Copyright 2002 Ariya Hidayat <ariya@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#include "KPrWebPresentation.h"

#include "KPrView.h"
#include "KPrDocument.h"
#include "KPrCanvas.h"
#include "KPrPage.h"

#include <tdestandarddirs.h>
#include <unistd.h>
#include <sys/types.h>
#include <tdetempfile.h>

#include <tqfile.h>
#include <tqtextstream.h>
#include <tqvbox.h>
#include <tqhbox.h>
#include <tqlabel.h>
#include <tqpushbutton.h>
#include <tqfileinfo.h>
#include <tqframe.h>
#include <tqfont.h>
#include <tqpixmap.h>
#include <tqdatetime.h>
#include <tqdir.h>
#include <tqheader.h>
#include <tqwmatrix.h>
#include <tqtextcodec.h>
#include <tqregexp.h>
#include <tqimage.h>
#include <tqlayout.h>
#include <tqwhatsthis.h>
#include <tqcheckbox.h>

#include <kdebug.h>
#include <tdelocale.h>
#include <kcolorbutton.h>
#include <tdefiledialog.h>
#include <tdemessagebox.h>
#include <kbuttonbox.h>
#include <tdesimpleconfig.h>
#include <tdeapplication.h>
#include <kprogress.h>
#include <tdeglobal.h>
#include <tdeglobalsettings.h>
#include <kcharsets.h>
#include <kurlrequester.h>
#include <klineedit.h>
#include <tdelistview.h>
#include <knuminput.h>
#include <kcombobox.h>
#include <kurl.h>
#include <tdeio/netaccess.h>
#include <kdialog.h>

#include "KoDocumentInfo.h"


// Comes from koffice/filters/libexport/KWEFUtils.cpp
static TQString EscapeSgmlText(const TQTextCodec* codec, const TQString& strIn,
                              const bool quot = false , const bool apos = false )
{
    TQString strReturn;
    TQChar ch;

    for (uint i=0; i<strIn.length(); i++)
    {
        ch=strIn[i];
        switch (ch.unicode())
        {
        case 38: // &
        {
            strReturn+="&amp;";
            break;
        }
        case 60: // <
        {
            strReturn+="&lt;";
            break;
        }
        case 62: // >
        {
            strReturn+="&gt;";
            break;
        }
        case 34: // "
        {
            if (quot)
                strReturn+="&quot;";
            else
                strReturn+=ch;
            break;
        }
        case 39: // '
        {
            // NOTE:  HTML does not define &apos; by default (only XML/XHTML does)
            if (apos)
                strReturn+="&apos;";
            else
                strReturn+=ch;
            break;
        }
        default:
        {
            // verify that the character ch can be expressed in the
            //   encoding in which we will write the HTML file.
            if (codec)
            {
                if (!codec->canEncode(ch))
                {
                    strReturn+=TQString("&#%1;").arg(ch.unicode());
                    break;
                }
            }
            strReturn+=ch;
            break;
        }
        }
    }

    return strReturn;
}

// Escape only if the encoding do not support the character
// Special SGML characters like < > & are supposed to be already escaped.
static TQString EscapeEncodingOnly(const TQTextCodec* codec, const TQString& strIn)
{
    TQString strReturn;
    TQChar ch;

    for (uint i=0; i<strIn.length(); i++)
    {
        ch=strIn[i];
        if (codec)
        {
            if (!codec->canEncode(ch))
            {
                strReturn+=TQString("&#%1;").arg(ch.unicode());
                continue;
            }
        }
        strReturn+=ch;
    }
    return strReturn;
}

KPrWebPresentation::KPrWebPresentation( KPrDocument *_doc, KPrView *_view )
    : config( TQString() ), xml( false )
{
    doc = _doc;
    view = _view;
    init();
}

KPrWebPresentation::KPrWebPresentation( const TQString &_config, KPrDocument *_doc, KPrView *_view )
    : config( _config ), xml( false ), m_bWriteHeader( true ), m_bWriteFooter( true ), m_bLoopSlides( false )
{
    doc = _doc;
    view = _view;
    init();
    loadConfig();
}

KPrWebPresentation::KPrWebPresentation( const KPrWebPresentation &webPres )
    : config( webPres.config ), author( webPres.author ), title( webPres.title ), email( webPres.email ),
      slideInfos( webPres.slideInfos ), backColor( webPres.backColor ), titleColor( webPres.titleColor ),
      textColor( webPres.textColor ), path( webPres.path ), xml( webPres.xml),
      m_bWriteHeader( webPres.m_bWriteHeader ),
      m_bWriteFooter( webPres.m_bWriteFooter ), m_bLoopSlides( webPres.m_bLoopSlides ),
      timeBetweenSlides ( webPres.timeBetweenSlides ), zoom( webPres.zoom ), m_encoding( webPres.m_encoding )
{
    doc = webPres.doc;
    view = webPres.view;
}

void KPrWebPresentation::loadConfig()
{
    if ( config.isEmpty() )
        return;

    TDESimpleConfig cfg( config );
    cfg.setGroup( "General" );

    author = cfg.readEntry( "Author", author );
    title = cfg.readEntry( "Title", title );
    email = cfg.readEntry( "EMail", email );
    unsigned int num = cfg.readNumEntry( "Slides", slideInfos.count() );
    //kdDebug(33001) << "KPrWebPresentation::loadConfig num=" << num << endl;

    if ( num <= slideInfos.count() ) {
        for ( unsigned int i = 0; i < num; i++ )
        {
            TQString key = TQString::fromLatin1( "SlideTitle%1" ).arg( i );
            if ( cfg.hasKey( key ) )
            {
                // We'll assume that the selected pages haven't changed... Hmm.
                slideInfos[ i ].slideTitle = cfg.readEntry( key );
                kdDebug(33001) << "KPrWebPresentation::loadConfig key=" << key << " data=" << slideInfos[i].slideTitle << endl;
            } else kdDebug(33001) << " key not found " << key << endl;
        }
    }

    backColor = cfg.readColorEntry( "BackColor", &backColor );
    titleColor = cfg.readColorEntry( "TitleColor", &titleColor );
    textColor = cfg.readColorEntry( "TextColor", &textColor );
    path = cfg.readPathEntry( "Path", path );
    xml = cfg.readBoolEntry( "XML", xml );
    m_bWriteHeader = cfg.readBoolEntry( "WriteHeader", m_bWriteHeader );
    m_bWriteFooter = cfg.readBoolEntry( "WriteFooter", m_bWriteFooter );
    m_bLoopSlides = cfg.readBoolEntry( "LoopSlides", m_bLoopSlides );
    zoom = cfg.readNumEntry( "Zoom", zoom );
    timeBetweenSlides = cfg.readNumEntry("TimeBetweenSlides", timeBetweenSlides );
    m_encoding = cfg.readEntry( "Encoding", m_encoding );
}

void KPrWebPresentation::saveConfig()
{
    TDESimpleConfig cfg( config );
    cfg.setGroup( "General" );

    cfg.writeEntry( "Author", author );
    cfg.writeEntry( "Title", title );
    cfg.writeEntry( "EMail", email );
    cfg.writeEntry( "Slides", slideInfos.count() );

    for ( unsigned int i = 0; i < slideInfos.count(); i++ )
        cfg.writeEntry( TQString::fromLatin1( "SlideTitle%1" ).arg( i ), slideInfos[ i ].slideTitle );

    cfg.writeEntry( "BackColor", backColor );
    cfg.writeEntry( "TitleColor", titleColor );
    cfg.writeEntry( "TextColor", textColor );
#if KDE_IS_VERSION(3,1,3)
    cfg.writePathEntry( "Path", path );
#else
    cfg.writeEntry( "Path", path );
#endif
    cfg.writeEntry( "XML", xml );
    cfg.writeEntry( "WriteHeader", m_bWriteHeader );
    cfg.writeEntry( "WriteFooter", m_bWriteFooter );
    cfg.writeEntry( "LoopSlides", m_bLoopSlides );
    cfg.writeEntry( "Zoom", zoom );
    cfg.writeEntry( "TimeBetweenSlides", timeBetweenSlides );
    cfg.writeEntry( "Encoding", m_encoding );
}

void KPrWebPresentation::initCreation( KProgress *progressBar )
{
    TQString cmd;
    int p;
    KURL str(  path + "/html"  );
    TDEIO::NetAccess::mkdir( str,( TQWidget* )0L  );

    p = progressBar->progress();
    progressBar->setProgress( ++p );
    tdeApp->processEvents();

    str = path + "/pics";
    TDEIO::NetAccess::mkdir( str,( TQWidget* )0L );

    p = progressBar->progress();
    progressBar->setProgress( ++p );
    tdeApp->processEvents();

    const char *pics[] = { "home", "first", "next", "prev", "last", 0 };

    KURL srcurl, desturl;

    for ( uint index = 0; pics[ index ]; index ++ )
    {
        TQString filename = pics[ index ];
        filename += ".png";
        srcurl.setPath( locate( "slideshow", filename, KPrFactory::global() ) );
        desturl = path;
        desturl.addPath( "/pics/" + filename );
        TDEIO::NetAccess::file_copy( srcurl, desturl, -1, true /*overwrite*/);
        p = progressBar->progress();
        progressBar->setProgress( ++p );
        tdeApp->processEvents();
    }
}

void KPrWebPresentation::createSlidesPictures( KProgress *progressBar )
{
    if ( slideInfos.isEmpty() )
        return;
    TQPixmap pix( 10, 10 );
    TQString filename;
    int p;
    for ( unsigned int i = 0; i < slideInfos.count(); i++ ) {
        int pgNum = slideInfos[i].pageNumber;
        view->getCanvas()->drawPageInPix( pix, pgNum, zoom, true /*force real variable value*/ );
        filename = TQString( "%1/pics/slide_%2.png" ).arg( path ).arg( i + 1 );

        KTempFile tmp;
        pix.save( tmp.name(), "PNG" );

        TDEIO::NetAccess::file_move( tmp.name(), filename, -1, true /*overwrite*/);

        p = progressBar->progress();
        progressBar->setProgress( ++p );
        tdeApp->processEvents();
    }
}

TQString KPrWebPresentation::escapeHtmlText( TQTextCodec *codec, const TQString& strText ) const
{
    // Escape quotes (needed in attributes)
    // Do not escape apostrophs (only allowed in XHTML!)
    return EscapeSgmlText( codec, strText, true, false );
}

void KPrWebPresentation::writeStartOfHeader(TQTextStream& streamOut, TQTextCodec *codec, const TQString& subtitle, const TQString& next)
{
    TQString mimeName ( codec->mimeName() );
    if ( isXML() )
    {   //Write out the XML declaration
        streamOut << "<?xml version=\"1.0\" encoding=\""
                  << mimeName << "\"?>\n";
    }
    // write <!DOCTYPE
    streamOut << "<!DOCTYPE ";
    if ( isXML() )
    {
        streamOut << "html PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\"";
        streamOut << " \"DTD/xhtml1-transitional.dtd\">\n";
    }
    else
    {
        streamOut << "HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\"";
        streamOut << " \"http://www.w3.org/TR/html4/loose.dtd\">\n";
    }
    streamOut << "<html";
    if ( isXML() )
    {
        // XHTML has an extra attribute defining its namespace (in the <html> opening tag)
        streamOut << " xmlns=\"http://www.w3.org/1999/xhtml\"";
    }
    streamOut << ">\n" << "<head>\n";

    // Declare what charset we are using
    streamOut << "<meta http-equiv=\"Content-Type\" content=\"text/html; charset=";
    streamOut << mimeName << '"' << ( isXML() ?" /":"") << ">\n" ;

    // Tell who we are (with the CVS revision number) in case we have a bug in our output!
    TQString strVersion("$Revision: 508787 $");
    // Eliminate the dollar signs
    //  (We don't want that the version number changes if the HTML file is itself put in a CVS storage.)
    streamOut << "<meta name=\"Generator\" content=\"KPresenter's Web Presentation "
              << strVersion.mid(10).replace("$","")
              << "\""<< ( isXML() ?" /":"") // X(HT)ML closes empty elements, HTML not!
              << ">\n";

    // Load the next slide after time elapsed
    if ( (timeBetweenSlides > 0) && ( ! next.isNull() ) )
    {
        streamOut << "<meta http-equiv=\"refresh\" content=\""
                  << timeBetweenSlides
                  << ";url=" << next
                  << "\">\n";
    }

    streamOut << "<title>"<< escapeHtmlText( codec, title ) << " - " << escapeHtmlText( codec, subtitle ) << "</title>\n";

    // ### TODO: transform documentinfo.xml into many <META> elements (at least the author!)
}

void KPrWebPresentation::createSlidesHTML( KProgress *progressBar )
{
    TQTextCodec *codec = TDEGlobal::charsets()->codecForName( m_encoding );

    const TQString brtag ( "<br" + TQString(isXML()?" /":"") + ">" );

    for ( unsigned int i = 0; i < slideInfos.count(); i++ ) {

        unsigned int pgNum = i + 1; // pgquiles # elpauer . org - I think this is a bug, seems to be an overflow if we have max_unsigned_int slides
        KTempFile tmp;
        TQString dest= TQString( "%1/html/slide_%2.html" ).arg( path ).arg( pgNum );
        TQString next= TQString( "slide_%2.html" ).arg( pgNum<slideInfos.count() ? pgNum+1 : (m_bLoopSlides ? 1 : pgNum ) ); // Ugly, but it works

        TQFile file( tmp.name() );
        file.open( IO_WriteOnly );
        TQTextStream streamOut( &file );
        streamOut.setCodec( codec );

        writeStartOfHeader( streamOut, codec, slideInfos[ i ].slideTitle, next );

        // ### TODO: transform documentinfo.xml into many <META> elements (at least the author!)

        if ( i > 0 ) {
            streamOut <<  "<link rel=\"first\" href=\"slide_1.html\"" << ( isXML() ?" /":"") << ">\n";
            streamOut <<  "<link rel=\"prev\" href=\"slide_" << pgNum - 1 << ".html\"" << ( isXML() ?" /":"") << ">\n";
        }
        if ( i < slideInfos.count() - 1 ) {
            streamOut <<  "<link rel=\"next\" href=\"slide_" << pgNum + 1 << ".html\"" << ( isXML() ?" /":"") << ">\n";
            streamOut <<  "<link rel=\"last\" href=\"slide_" << slideInfos.count() << ".html\"" << ( isXML() ?" /":"") << ">\n";
        }
        streamOut <<  "<link rel=\"contents\" href=\"../index.html\"" << ( isXML() ?" /":"") << ">\n";

        streamOut << "</head>\n";
        streamOut << "<body bgcolor=\"" << backColor.name() << "\" text=\"" << textColor.name() << "\">\n";

        if (m_bWriteHeader) {
            streamOut << "  <center>\n";

            if ( i > 0 )
                streamOut << "    <a href=\"slide_1.html\">";
                streamOut << "<img src=\"../pics/first.png\" border=\"0\" alt=\"" << i18n( "First" )
                              << "\" title=\"" << i18n( "First" ) << "\"" << ( isXML() ?" /":"") << ">";
            if ( i > 0 )
                streamOut << "</a>";

            streamOut << "\n";

            if ( i > 0 )
                streamOut << "    <a href=\"slide_" << pgNum - 1 << ".html\">";
                streamOut << "<img src=\"../pics/prev.png\" border=\"0\" alt=\"" << i18n( "Previous" )
                              << "\" title=\"" << i18n( "Previous" ) << "\"" << ( isXML() ?" /":"") << ">";
            if ( i > 0 )
                streamOut << "</a>";

            streamOut << "\n";

            if ( (m_bLoopSlides) || (i < slideInfos.count() - 1 ) )
                streamOut << "    <a href=\"" << next << "\">";
                streamOut << "<img src=\"../pics/next.png\" border=\"0\" alt=\"" << i18n( "Next" )
                          << "\" title=\"" << i18n( "Next" ) << "\"" << ( isXML() ?" /":"") << ">";
            if ( (m_bLoopSlides) || (i < slideInfos.count() - 1 ) )
                streamOut << "</a>";

            streamOut << "\n";

            if ( i < slideInfos.count() - 1 )
                streamOut << "    <a href=\"slide_" << slideInfos.count() << ".html\">";
                streamOut << "<img src=\"../pics/last.png\" border=\"0\" alt=\"" << i18n( "Last" )
                      << "\" title=\"" << i18n( "Last" ) << "\"" << ( isXML() ?" /":"") << ">";
            if ( i < slideInfos.count() - 1 )
                streamOut << "</a>";

            streamOut << "\n" << "    &nbsp; &nbsp; &nbsp; &nbsp;\n";

            streamOut << "    <a href=\"../index.html\">";
            streamOut << "<img src=\"../pics/home.png\" border=\"0\" alt=\"" << i18n( "Home" )
                      << "\" title=\"" << i18n( "Home" ) << "\"" << ( isXML() ?" /":"") << ">";
            streamOut << "</a>\n";

            streamOut << " </center>" << brtag << "<hr noshade=\"noshade\"" << ( isXML() ?" /":"") << ">\n"; // ### TODO: is noshade W3C?

            streamOut << "  <center>\n    <font color=\"" << escapeHtmlText( codec, titleColor.name() ) << "\">\n";
            streamOut << "    <b>" << escapeHtmlText( codec, title ) << "</b> - <i>" << escapeHtmlText( codec, slideInfos[ i ].slideTitle ) << "</i>\n";

            streamOut << "    </font>\n  </center>\n";

            streamOut << "<hr noshade=\"noshade\"" << ( isXML() ?" /":"") << ">" << brtag << "\n";
	}

        streamOut << "  <center>\n    ";

	if ( (m_bLoopSlides) || (i < slideInfos.count() - 1) )
            streamOut << "<a href=\"" << next << "\">";

	    streamOut << "<img src=\"../pics/slide_" << pgNum << ".png\" border=\"0\" alt=\""
                      << i18n( "Slide %1" ).arg( pgNum ) << "\"" << ( isXML() ?" /":"") << ">";

	    if ( i < slideInfos.count() - 1 )
                streamOut << "</a>";

            streamOut << "\n";

            streamOut << "    </center>\n";

	if (m_bWriteFooter) {
	    	streamOut << brtag << "<hr noshade=\"noshade\"" << ( isXML() ?" /":"") << ">\n";

            TQPtrList<KPrPage> _tmpList( doc->getPageList() );
            TQString note ( escapeHtmlText( codec, _tmpList.at(i)->noteText() ) );
            if ( !note.isEmpty() ) {
                streamOut << "  <b>" << escapeHtmlText( codec, i18n( "Note" ) ) << "</b>\n";
                streamOut << " <blockquote>\n";

                streamOut << note.replace( "\n", brtag );

                streamOut << "  </blockquote><hr noshade=\"noshade\"" << ( isXML() ?" /":"") << ">\n";
            }

            streamOut << "  <center>\n";

            TQString htmlAuthor;
            if (email.isEmpty())
                htmlAuthor=escapeHtmlText( codec, author );
            else
                htmlAuthor=TQString("<a href=\"mailto:%1\">%2</a>").arg( escapeHtmlText( codec, email )).arg( escapeHtmlText( codec, author ));
            streamOut << EscapeEncodingOnly ( codec, i18n( "Created on %1 by <i>%2</i> with <a href=\"http://www.trinitydesktop.org\">KPresenter</a>" )
                                          .arg( TDEGlobal::locale()->formatDate ( TQDate::currentDate() ) ).arg( htmlAuthor ) );

            streamOut << "    </center><hr noshade=\"noshade\"" << ( isXML() ?" /":"") << ">\n";
        }

        streamOut << "</body>\n</html>\n";

        file.close();

        TDEIO::NetAccess::file_move( tmp.name(), dest, -1, true /*overwrite*/);

        int p = progressBar->progress();
        progressBar->setProgress( ++p );
        tdeApp->processEvents();
    }
}

void KPrWebPresentation::createMainPage( KProgress *progressBar )
{
    TQTextCodec *codec = TDEGlobal::charsets()->codecForName( m_encoding );
    KTempFile tmp;
    TQString dest = TQString( "%1/index.html" ).arg( path );
    TQFile file( tmp.name() );
    file.open( IO_WriteOnly );
    TQTextStream streamOut( &file );
    streamOut.setCodec( codec );

    writeStartOfHeader( streamOut, codec, i18n("Table of Contents"), TQString() );
    streamOut << "</head>\n";

    streamOut << "<body bgcolor=\"" << backColor.name() << "\" text=\"" << textColor.name() << "\">\n";

    streamOut << "<h1 align=\"center\"><font color=\"" << titleColor.name()
              << "\">" << title << "</font></h1>";

    streamOut << "<p align=\"center\"><a href=\"html/slide_1.html\">";
    streamOut << i18n("Click here to start the Slideshow");
    streamOut << "</a></p>\n";

    streamOut << "<p><b>" << i18n("Table of Contents") << "</b></p>\n";

    // create list of slides (with proper link)
    streamOut << "<ol>\n";
    for ( unsigned int i = 0; i < slideInfos.count(); i++ )
        streamOut << "  <li><a href=\"html/slide_" << i+1 << ".html\">" << slideInfos[ i ].slideTitle << "</a></li>\n";
    streamOut << "</ol>\n";

    // footer: author name, e-mail
    TQString htmlAuthor = email.isEmpty() ? escapeHtmlText( codec, author ) :
                         TQString("<a href=\"mailto:%1\">%2</a>").arg( escapeHtmlText( codec, email )).arg( escapeHtmlText( codec, author ));
    streamOut << EscapeEncodingOnly ( codec, i18n( "Created on %1 by <i>%2</i> with <a href=\"http://www.trinitydesktop.org\">KPresenter</a>" )
                                      .arg( TDEGlobal::locale()->formatDate ( TQDate::currentDate() ) ).arg( htmlAuthor ) );

    streamOut << "</body>\n</html>\n";
    file.close();

    TDEIO::NetAccess::file_move( tmp.name(), dest, -1, true /*overwrite*/);


    progressBar->setProgress( progressBar->totalSteps() );
    tdeApp->processEvents();
}

void KPrWebPresentation::init()
{

    KoDocumentInfo * info = doc->documentInfo();
    KoDocumentInfoAuthor * authorPage = static_cast<KoDocumentInfoAuthor *>(info->page( "author" ));
    if ( !authorPage )
        kdWarning() << "Author information not found in documentInfo !" << endl;
    else
    {
        author = authorPage->fullName();
        email = authorPage->email();
    }

    title = i18n("Slideshow");
    kdDebug(33001) << "KPrWebPresentation::init : " << doc->getPageNums() << " pages." << endl;
    for ( unsigned int i = 0; i < doc->getPageNums(); i++ )
    {
        if ( doc->isSlideSelected( i ) )
        {
            SlideInfo info;
            info.pageNumber = i;
            info.slideTitle = doc->pageList().at(i)->pageTitle();
            slideInfos.append( info );
        }
    }
    if ( slideInfos.isEmpty() )
        kdWarning() << "No slides selected!" << endl;
    backColor = TQt::white;
    textColor = TQt::black;
    titleColor = TQt::red;

    path = TDEGlobalSettings::documentPath() + "www";

    zoom = 100;

    timeBetweenSlides = 0;

    m_encoding = TQTextCodec::codecForLocale()->name();
}

KPrWebPresentationWizard::KPrWebPresentationWizard( const TQString &_config, KPrDocument *_doc,
                                                  KPrView *_view )
    : KWizard( 0, "", false ), config( _config ), webPres( config, _doc, _view )
{
    doc = _doc;
    view = _view;

    setupPage1();
    setupPage2();
    setupPage3();
    setupPage4();
    setupPage5();

    connect( nextButton(), TQ_SIGNAL( clicked() ), this, TQ_SLOT( pageChanged() ) );
    connect( backButton(), TQ_SIGNAL( clicked() ), this, TQ_SLOT( pageChanged() ) );
    connect( finishButton(), TQ_SIGNAL( clicked() ), this, TQ_SLOT( finish() ) );
}

KPrWebPresentationWizard::~KPrWebPresentationWizard()
{
    view->enableWebPres();
}

void KPrWebPresentationWizard::createWebPresentation( const TQString &_config, KPrDocument *_doc,
                                                     KPrView *_view )
{
    KPrWebPresentationWizard *dlg = new KPrWebPresentationWizard( _config, _doc, _view );

    dlg->setCaption( i18n( "Create HTML Slideshow Wizard" ) );
    dlg->show();
}

void KPrWebPresentationWizard::setupPage1()
{
    page1 = new TQHBox( this );
    TQWhatsThis::add( page1, i18n("This page allows you to specify some of the key"
                                 " values for how your presentation will be shown"
                                 " in HTML. Select individual items for more help"
                                 " on what they do.") );
    page1->setSpacing( KDialog::spacingHint() );
    page1->setMargin( KDialog::marginHint() );

    TQLabel* sidebar = new TQLabel( page1 );
    sidebar->setMinimumSize( 106, 318 );
    sidebar->setMaximumSize( 106, 318 );
    sidebar->setFrameShape( TQFrame::Panel );
    sidebar->setFrameShadow( TQFrame::Sunken );
    sidebar->setPixmap(locate("data", "kpresenter/pics/webslideshow-sidebar.png"));

    TQWidget* canvas = new TQWidget( page1 );
    TQGridLayout *layout = new TQGridLayout( canvas, 7, 2,
                                           KDialog::marginHint(), KDialog::spacingHint() );

    TQLabel *helptext = new TQLabel( canvas );
    helptext->setAlignment( TQt::WordBreak | TQt::AlignTop| TQt::AlignLeft );
    helptext->setText( i18n( "Enter your name, email address and "
                             "the title of the web presentation. "
                             "Also enter the output directory where the "
                             "web presentation should be saved. " ) );
    layout->addMultiCellWidget( helptext, 0, 0, 0, 1 );

    layout->addMultiCell( new TQSpacerItem( 1, 50 ), 1, 1, 0, 1 );

    TQLabel *label1 = new TQLabel( i18n("Author:"), canvas );
    label1->setAlignment( TQt::AlignVCenter | TQt::AlignRight );
    TQWhatsThis::add( label1, i18n("This is where you enter the name of the person or "
                                  "organization that should be named as the author of "
                                  "the presentation.") );
    layout->addWidget( label1, 2, 0 );

    TQLabel *label2 = new TQLabel( i18n("Title:"), canvas );
    label2->setAlignment( TQt::AlignVCenter | TQt::AlignRight );
    TQWhatsThis::add( label2, i18n("This is where you enter the title of the overall "
                                  "presentation." ) );
    layout->addWidget( label2, 3, 0 );

    TQLabel *label3 = new TQLabel( i18n("Email address:"), canvas );
    label3->setAlignment( TQt::AlignVCenter | TQt::AlignRight );
    TQWhatsThis::add( label3, i18n("This is where you enter the email address of the "
                                  "person or organization that is responsible for "
                                  "the presentation.") );
    layout->addWidget( label3, 4, 0 );

    TQLabel *label4 = new TQLabel( i18n("Path:"), canvas );
    label4->setAlignment( TQt::AlignVCenter | TQt::AlignRight );
    TQWhatsThis::add( label4, i18n("The value entered for the path is the directory "
                                  "where the presentation will be saved. If it does "
                                  "not exist, you'll be asked if you want to create "
                                  "the directory or abort the creation.") );
    layout->addWidget( label4, 5, 0 );

    author = new KLineEdit( webPres.getAuthor(), canvas );
    TQWhatsThis::add( author, i18n("This is where you enter the name of the person or "
                                  "organization that should be named as the author of "
                                  "the presentation.") );
    layout->addWidget( author, 2, 1 );

    title = new KLineEdit( webPres.getTitle(), canvas );
    TQWhatsThis::add( title, i18n("This is where you enter the title of the overall "
                                 "presentation." ) );
    layout->addWidget( title, 3, 1 );

    email = new KLineEdit( webPres.getEmail(), canvas );
    TQWhatsThis::add( email, i18n("This is where you enter the email address of the "
                                 "person or organization that is responsible for "
                                 "the presentation.") );
    layout->addWidget( email, 4, 1 );

    path=new KURLRequester( canvas );
    path->setMode( KFile::Directory);
    path->lineEdit()->setText(webPres.getPath());
    TQWhatsThis::add( path, i18n("The value entered for the path is the directory "
                                "where the presentation will be saved. If it does "
                                "not exist, you'll be asked if you want to create "
                                "the directory or abort the creation.") );
    layout->addWidget( path, 5, 1 );

    TQSpacerItem* spacer = new TQSpacerItem( 1, 10,
                                           TQSizePolicy::Minimum, TQSizePolicy::Expanding );
    layout->addMultiCell( spacer, 6, 6, 0, 1 );

    connect(path, TQ_SIGNAL(textChanged(const TQString&)),
            this,TQ_SLOT(slotChoosePath(const TQString&)));
    connect(path, TQ_SIGNAL(urlSelected( const TQString& )),
            this,TQ_SLOT(slotChoosePath(const TQString&)));

    addPage( page1, i18n( "Step 1: General Information" ) );

    setHelpEnabled(page1, false);  //doesn't do anything currently
}

void KPrWebPresentationWizard::setupPage2()
{
    page2 = new TQHBox( this );
    TQWhatsThis::add( page2, i18n("This page allows you to specify how the HTML "
                                 "for your presentation will be displayed. Select "
                                 "individual items for more help on what they do.") );
    page2->setSpacing( KDialog::spacingHint() );
    page2->setMargin( KDialog::marginHint() );

    TQLabel* sidebar = new TQLabel( page2 );
    sidebar->setMinimumSize( 106, 318 );
    sidebar->setMaximumSize( 106, 318 );
    sidebar->setFrameShape( TQFrame::Panel );
    sidebar->setFrameShadow( TQFrame::Sunken );
    sidebar->setPixmap(locate("data", "kpresenter/pics/webslideshow-sidebar.png"));

    TQWidget* canvas = new TQWidget( page2 );
    TQGridLayout *layout = new TQGridLayout( canvas, 6, 2,
                                           KDialog::marginHint(), KDialog::spacingHint() );

    TQLabel *helptext = new TQLabel( canvas );
    helptext->setAlignment( TQt::WordBreak | TQt::AlignVCenter| TQt::AlignLeft );
    TQString help = i18n("Here you can configure the style of the web pages.");
    help += i18n( "You can also specify the zoom for the slides." );
    helptext->setText(help);

    layout->addMultiCellWidget( helptext, 0, 0, 0, 1 );

    layout->addMultiCell( new TQSpacerItem( 1, 50 ), 1, 1, 0, 1 );

    TQLabel *label1 = new TQLabel( i18n("Zoom:"), canvas );
    label1->setAlignment( TQt::AlignVCenter | TQt::AlignRight );
    TQWhatsThis::add( label1, i18n( "This selection allows you to specify "
                                   "the size of the slide image." ) );
    layout->addWidget( label1, 2, 0 );

    TQLabel *label2 = new TQLabel( i18n( "Encoding:" ), canvas );
    label2->setAlignment( TQt::AlignVCenter | TQt::AlignRight );

    layout->addWidget( label2, 3, 0 );

    TQLabel *label3 = new TQLabel( i18n( "Document type:" ), canvas );
    label3->setAlignment( TQt::AlignVCenter | TQt::AlignRight );
    layout->addWidget( label3, 4, 0 );

    zoom = new KIntNumInput( webPres.getZoom(), canvas );
    TQWhatsThis::add( zoom, i18n( "This selection allows you to specify "
                                 "the size of the slide image." ) );
    layout->addWidget( zoom, 2, 1 );
    zoom->setSuffix( " %" );
    zoom->setRange( 25, 1000, 5 );

    encoding = new KComboBox( false, canvas );
    layout->addWidget( encoding, 3, 1 );

    // Fill encoding combo
    // Stolen from tdelibs/kate/part/katedialogs.cpp
    TQStringList encodings(TDEGlobal::charsets()->descriptiveEncodingNames());
    int idx = 0;
    for (uint i = 0; i < encodings.count(); i++)
    {
      bool found = false;
      TQTextCodec *codecForEnc = TDEGlobal::charsets()->codecForName(TDEGlobal::charsets()->encodingForName(encodings[i]), found);
      if (found)
      {
        encoding->insertItem(encodings[i]);

        if ( codecForEnc->name() == webPres.getEncoding() )
          encoding->setCurrentItem(idx);
        idx++;
      }
    }

    doctype = new KComboBox( false, canvas );
    layout->addWidget( doctype, 4, 1 );
    doctype->insertItem( "HTML 4.01", -1 );
    doctype->insertItem( "XHTML 1.0", -1 );

    doctype->setCurrentItem( webPres.isXML() ? 1 : 0 );

    TQSpacerItem* spacer = new TQSpacerItem( 1, 10,
                                           TQSizePolicy::Minimum, TQSizePolicy::Expanding );
    layout->addMultiCell( spacer, 5, 5, 0, 1 );

    addPage( page2, i18n( "Step 2: Configure HTML" ) );

    setHelpEnabled(page2, false);  //doesn't do anything currently
}

void KPrWebPresentationWizard::setupPage3()
{
    page3 = new TQHBox( this );
    TQWhatsThis::add( page3, i18n("This page allows you to specify the colors for "
                                 "your presentation display. Select individual "
                                 "items for more help on what they do.") );

    page3->setSpacing( KDialog::spacingHint() );
    page3->setMargin( KDialog::marginHint() );

    TQLabel* sidebar = new TQLabel( page3 );
    sidebar->setMinimumSize( 106, 318 );
    sidebar->setMaximumSize( 106, 318 );
    sidebar->setFrameShape( TQFrame::Panel );
    sidebar->setFrameShadow( TQFrame::Sunken );
    sidebar->setPixmap(locate("data", "kpresenter/pics/webslideshow-sidebar.png"));

    TQWidget* canvas = new TQWidget( page3 );
    TQGridLayout *layout = new TQGridLayout( canvas, 6, 2,
                                           KDialog::marginHint(), KDialog::spacingHint() );

    TQLabel *helptext = new TQLabel( canvas );
    helptext->setAlignment( TQt::WordBreak | TQt::AlignVCenter| TQt::AlignLeft );
    helptext->setText( i18n( "Now you can customize the colors of the web pages." ) );
    layout->addMultiCellWidget( helptext, 0, 0, 0, 1 );

    layout->addMultiCell( new TQSpacerItem( 1, 50 ), 1, 1, 0, 1 );

    TQLabel *label1 = new TQLabel( i18n("Text color:"), canvas );
    label1->setAlignment( TQt::AlignVCenter | TQt::AlignRight );
    layout->addWidget( label1, 2, 0 );

    TQLabel *label2 = new TQLabel( i18n("Title color:"), canvas );
    label2->setAlignment( TQt::AlignVCenter | TQt::AlignRight );
    layout->addWidget( label2, 3, 0 );

    TQLabel *label3 = new TQLabel( i18n("Background color:"), canvas );
    label3->setAlignment( TQt::AlignVCenter | TQt::AlignRight );
    layout->addWidget( label3, 4, 0 );

    textColor = new KColorButton( webPres.getTextColor(), canvas );
    layout->addWidget( textColor, 2, 1 );

    titleColor = new KColorButton( webPres.getTitleColor(), canvas );
    layout->addWidget( titleColor, 3, 1 );

    backColor = new KColorButton( webPres.getBackColor(), canvas );
    layout->addWidget( backColor, 4, 1 );

    TQSpacerItem* spacer = new TQSpacerItem( 1, 10,
                                           TQSizePolicy::Minimum, TQSizePolicy::Expanding );
    layout->addMultiCell( spacer, 5, 5, 0, 1 );

    addPage( page3, i18n( "Step 3: Customize Colors" ) );

    setHelpEnabled(page3, false);  //doesn't do anything currently
}

void KPrWebPresentationWizard::setupPage4()
{
    page4 = new TQHBox( this );
    TQWhatsThis::add( page4, i18n("This page allows you to modify the titles of "
                                 "each slide, if required. You normally do not need "
                                 "to do this, but it is available if required.") );
    page4->setSpacing( KDialog::spacingHint() );
    page4->setMargin( KDialog::marginHint() );

    TQLabel* sidebar = new TQLabel( page4 );
    sidebar->setMinimumSize( 106, 318 );
    sidebar->setMaximumSize( 106, 318 );
    sidebar->setFrameShape( TQFrame::Panel );
    sidebar->setFrameShadow( TQFrame::Sunken );
    sidebar->setPixmap(locate("data", "kpresenter/pics/webslideshow-sidebar.png"));

    TQWidget* canvas = new TQWidget( page4 );
    TQGridLayout *layout = new TQGridLayout( canvas, 3, 2,
                                           KDialog::marginHint(), KDialog::spacingHint() );

    TQLabel *helptext = new TQLabel( canvas );
    helptext->setAlignment( TQt::WordBreak | TQt::AlignVCenter| TQt::AlignLeft );
    helptext->setText( i18n( "Here you can specify titles for "
                             "each slide. Click on a slide in "
                             "the list and then enter the title "
                             "in the textbox below. If you "
                             "click on a title, KPresenter "
                             "mainview will display the slide.") );

    layout->addMultiCellWidget( helptext, 0, 0, 0, 1 );

    TQLabel *label = new TQLabel( i18n( "Slide title:" ), canvas );
    label->setAlignment( TQt::AlignVCenter | TQt::AlignRight );
    layout->addWidget( label, 1, 0 );

    slideTitle = new KLineEdit( canvas );
    layout->addWidget( slideTitle, 1, 1 );
    connect( slideTitle, TQ_SIGNAL( textChanged( const TQString & ) ), this,
             TQ_SLOT( slideTitleChanged( const TQString & ) ) );

    slideTitles = new TDEListView( canvas );
    layout->addMultiCellWidget( slideTitles, 2, 2, 0, 1 );
    slideTitles->addColumn( i18n( "No." ) );
    slideTitles->addColumn( i18n( "Slide Title" ) );
    connect( slideTitles, TQ_SIGNAL( selectionChanged( TQListViewItem * ) ), this,
             TQ_SLOT( slideTitleChanged( TQListViewItem * ) ) );
    slideTitles->setSorting( -1 );
    slideTitles->setAllColumnsShowFocus( true );
    slideTitles->setResizeMode( TQListView::LastColumn );
    slideTitles->header()->setMovingEnabled( false );

    TQValueList<KPrWebPresentation::SlideInfo> infos = webPres.getSlideInfos();
    for ( int i = infos.count() - 1; i >= 0; --i ) {
        TDEListViewItem *item = new TDEListViewItem( slideTitles );
        item->setText( 0, TQString::number( i + 1 ) );
        //kdDebug(33001) << "KPrWebPresentationWizard::setupPage3 " << infos[ i ].slideTitle << endl;
        item->setText( 1, infos[ i ].slideTitle );
    }

    slideTitles->setSelected( slideTitles->firstChild(), true );

    addPage( page4, i18n( "Step 4: Customize Slide Titles" ) );

    setHelpEnabled(page4, false);  //doesn't do anything currently
}

void KPrWebPresentationWizard::setupPage5()
{
    page5 = new TQHBox( this );
    TQWhatsThis::add( page5, i18n("This page allows you to specify some options for "
                                 "presentations which run unattended, such as time "
                                 "elapsed before advancing to the next slide, looping "
                                 "and the presence of headers. If you do not want "
                                 "an unattended presentation, just leave defaults unchanged.") );
    page5->setSpacing( KDialog::spacingHint() );
    page5->setMargin( KDialog::marginHint() );

    TQLabel* sidebar = new TQLabel( page5 );
    sidebar->setMinimumSize( 106, 318 );
    sidebar->setMaximumSize( 106, 318 );
    sidebar->setFrameShape( TQFrame::Panel );
    sidebar->setFrameShadow( TQFrame::Sunken );
    sidebar->setPixmap(locate("data", "kpresenter/pics/webslideshow-sidebar.png"));

    TQWidget* canvas = new TQWidget( page5 );
    TQGridLayout *layout = new TQGridLayout( canvas, 6, 2,
                                           KDialog::marginHint(), KDialog::spacingHint() );

    TQLabel *helptext = new TQLabel( canvas );
    helptext->setAlignment( TQt::WordBreak | TQt::AlignVCenter| TQt::AlignLeft );
    TQString help = i18n("Here you can configure some options for unattended "
                        "presentations, such as time elapsed before automatically advance to "
                        "the next slide, looping and the presence of headers.");
    helptext->setText(help);

    layout->addMultiCellWidget( helptext, 0, 0, 0, 1 );

    layout->addMultiCell( new TQSpacerItem( 1, 50 ), 1, 1, 0, 1 );

    TQLabel *label1 = new TQLabel( i18n("Advance after:"), canvas );
    label1->setAlignment( TQt::AlignVCenter | TQt::AlignRight );
    TQWhatsThis::add( label1, i18n( "This selection allows you to specify "
                                   "the time between slides." ) );
    layout->addWidget( label1, 2, 0 );

    timeBetweenSlides = new KIntNumInput( webPres.getTimeBetweenSlides(), canvas );
    timeBetweenSlides->setSpecialValueText(i18n( "Disabled" ));
    TQWhatsThis::add( timeBetweenSlides, i18n( "This selection allows you to specify "
                                 "the time between slides." ) );
    layout->addWidget( timeBetweenSlides, 2, 1 );
    timeBetweenSlides->setSuffix( " seconds" );
    timeBetweenSlides->setRange( 0, 900, 1 );

    layout->addMultiCell( new TQSpacerItem( 1, 10 ), 1, 1, 0, 1 );

    writeHeader=new TQCheckBox( i18n("Write header to the slides"), canvas);
    TQWhatsThis::add( writeHeader, i18n( "This checkbox allows you to specify if you "
                                       "want to write the navigation buttons on top "
                                       "of the slide." ) );
    writeHeader->setChecked( webPres.wantHeader() );
    layout->addWidget( writeHeader, 3, 1);

    writeFooter=new TQCheckBox( i18n("Write footer to the slides"), canvas);
    TQWhatsThis::add( writeFooter, i18n( "This checkbox allows you to specify if you "
                                       "want to write an imprint consisting on the author "
                                       "and the software used to create these slides." ) );
    writeFooter->setChecked( webPres.wantFooter() );
    layout->addWidget( writeFooter, 4, 1);

    loopSlides=new TQCheckBox( i18n("Loop presentation"), canvas);
    TQWhatsThis::add( loopSlides, i18n( "This checkbox allows you to specify if you "
                                       "want the presentation to start again once "
                                       "the latest slide is reached." ) );
    loopSlides->setChecked( webPres.wantLoopSlides() );
    layout->addWidget( loopSlides, 5, 1);

    TQSpacerItem* spacer = new TQSpacerItem( 1, 10,
                                           TQSizePolicy::Minimum, TQSizePolicy::Expanding );
    layout->addMultiCell( spacer, 5, 5, 0, 1 );

    addPage( page5, i18n( "Step 5: Options for Unattended Presentations" ) );

    setHelpEnabled(page5, false);  //doesn't do anything currently

    setFinish( page5, true );
}

void KPrWebPresentationWizard::finish()
{
    webPres.setAuthor( author->text() );
    webPres.setEMail( email->text() );
    webPres.setTitle( title->text() );

    TQListViewItemIterator it( slideTitles );
    for ( ; it.current(); ++it )
        webPres.setSlideTitle( it.current()->text( 0 ).toInt() - 1, it.current()->text( 1 ) );

    webPres.setBackColor( backColor->color() );
    webPres.setTitleColor( titleColor->color() );
    webPres.setTextColor( textColor->color() );
    webPres.setPath( path->lineEdit()->text() );
    webPres.setZoom( zoom->value() );
    webPres.setTimeBetweenSlides( timeBetweenSlides->value() );
    webPres.setWriteHeader( writeHeader->isChecked() );
    webPres.setWriteFooter( writeFooter->isChecked() );
    webPres.setLoopSlides( loopSlides->isChecked() );
    webPres.setXML( doctype->currentItem() != 0 );
    bool found = false;
    TQTextCodec *codecForEnc = TDEGlobal::charsets()->codecForName(TDEGlobal::charsets()->encodingForName(encoding->currentText()), found);
    if ( found )
    {
        webPres.setEncoding( codecForEnc->name() );
    }

    close();
    KPrWebPresentationCreateDialog::createWebPresentation( doc, view, webPres );
}

void KPrWebPresentationWizard::pageChanged()
{
    if ( currentPage() != page5 )
    {
        TQString pathname = path->lineEdit()->text();

        // path doesn't exist. ask user if it should be created.
        if ( !TDEIO::NetAccess::exists( pathname, true/*write*/,this ) )
        {
            TQString msg = i18n( "<qt>The directory <b>%1</b> does not exist.<br>"
                                "Do you want create it?</qt>" );
            if( KMessageBox::questionYesNo( this, msg.arg( pathname ),
                                            i18n( "Directory Not Found" ) )
                == KMessageBox::Yes)
            {
                bool ok = TDEIO::NetAccess::mkdir( pathname, this );
                if( !ok )
                {
                    KMessageBox::sorry( this,
                                        i18n( "Cannot create directory." ) );
                    // go back to first step
                    showPage( page1 );
                    path->setFocus();
                }

            }
            else
            {
                // go back to first step
                showPage( page1 );
                path->setFocus();
            }
        }
    } else
        finishButton()->setEnabled( true );
}

void KPrWebPresentationWizard::slideTitleChanged( const TQString &s )
{
    if ( slideTitles->currentItem() )
        slideTitles->currentItem()->setText( 1, s );
}

void KPrWebPresentationWizard::slideTitleChanged( TQListViewItem *i )
{
    if ( !i ) return;

    slideTitle->setText( i->text( 1 ) );
    view->skipToPage( i->text( 0 ).toInt() - 1 );
}

void KPrWebPresentationWizard::closeEvent( TQCloseEvent *e )
{
    view->enableWebPres();
    KWizard::closeEvent( e );
}

void KPrWebPresentationWizard::slotChoosePath(const TQString &text)
{
    webPres.setPath(text);
}

KPrWebPresentationCreateDialog::KPrWebPresentationCreateDialog( KPrDocument *_doc, KPrView *_view,
                                                              const KPrWebPresentation &_webPres )
    : TQDialog( 0, "", false ), webPres( _webPres )
{
    doc = _doc;
    view = _view;

    setupGUI();
}

KPrWebPresentationCreateDialog::~KPrWebPresentationCreateDialog()
{
    view->enableWebPres();
}

void KPrWebPresentationCreateDialog::createWebPresentation( KPrDocument *_doc, KPrView *_view,
                                                           const KPrWebPresentation &_webPres )
{
    KPrWebPresentationCreateDialog *dlg = new KPrWebPresentationCreateDialog( _doc, _view, _webPres );

    dlg->setCaption( i18n( "Create HTML Slideshow" ) );
    dlg->resize( 400, 300 );
    dlg->show();
    dlg->start();
}

void KPrWebPresentationCreateDialog::start()
{
    setCursor( waitCursor );
    initCreation();
    createSlidesPictures();
    createSlidesHTML();
    createMainPage();
    setCursor( arrowCursor );

    bDone->setEnabled( true );
    bSave->setEnabled( true );
}

void KPrWebPresentationCreateDialog::initCreation()
{
    TQFont f = step1->font(), f2 = step1->font();
    f.setBold( true );
    step1->setFont( f );

    progressBar->setProgress( 0 );
    progressBar->setTotalSteps( webPres.initSteps() );

    webPres.initCreation( progressBar );

    step1->setFont( f2 );
    progressBar->setProgress( progressBar->totalSteps() );
}

void KPrWebPresentationCreateDialog::createSlidesPictures()
{
    TQFont f = step2->font(), f2 = f;
    f.setBold( true );
    step2->setFont( f );

    progressBar->setProgress( 0 );
    if ( webPres.slides1Steps() > 0 )
    {
        progressBar->setTotalSteps( webPres.slides1Steps() );
        webPres.createSlidesPictures( progressBar );
    }

    step2->setFont( f2 );
    progressBar->setProgress( progressBar->totalSteps() );
}

void KPrWebPresentationCreateDialog::createSlidesHTML()
{
    TQFont f = step3->font(), f2 = step3->font();
    f.setBold( true );
    step3->setFont( f );

    progressBar->setProgress( 0 );
    if ( webPres.slides1Steps() > 0 )
    {
        progressBar->setTotalSteps( webPres.slides1Steps() );
        webPres.createSlidesHTML( progressBar );
    }

    step3->setFont( f2 );
    progressBar->setProgress( progressBar->totalSteps() );
}

void KPrWebPresentationCreateDialog::createMainPage()
{
    TQFont f = step4->font(), f2 = step4->font();
    f.setBold( true );
    step4->setFont( f );

    progressBar->setProgress( 0 );
    progressBar->setTotalSteps( webPres.slides1Steps() );

    webPres.createMainPage( progressBar );

    step4->setFont( f2 );
    progressBar->setProgress( progressBar->totalSteps() );
}

void KPrWebPresentationCreateDialog::setupGUI()
{
    back = new TQVBox( this );
    back->setMargin( KDialog::marginHint() );

    TQFrame *line;

    line = new TQFrame( back );
    line->setFrameStyle( TQFrame::HLine | TQFrame::Sunken );
    line->setMaximumHeight( 20 );

    step1 = new TQLabel( i18n( "Initialize (create file structure, etc.)" ), back );
    step2 = new TQLabel( i18n( "Create Pictures of the Slides" ), back );
    step3 = new TQLabel( i18n( "Create HTML Pages for the Slides" ), back );
    step4 = new TQLabel( i18n( "Create Main Page (Table of Contents)" ), back );
    step5 = new TQLabel( i18n( "Options for Unattended Presentations" ), back);

    line = new TQFrame( back );
    line->setFrameStyle( TQFrame::HLine | TQFrame::Sunken );
    line->setMaximumHeight( 20 );

    progressBar = new KProgress( back );

    line = new TQFrame( back );
    line->setFrameStyle( TQFrame::HLine | TQFrame::Sunken );
    line->setMaximumHeight( 20 );

    KButtonBox *bb = new KButtonBox( back );
    bSave = bb->addButton( i18n( "Save Configuration..." ) );
    bb->addStretch();
    bDone = bb->addButton( i18n( "Done" ) );

    bSave->setEnabled( false );
    bDone->setEnabled( false );

    connect( bDone, TQ_SIGNAL( clicked() ), this, TQ_SLOT( accept() ) );
    connect( bSave, TQ_SIGNAL( clicked() ), this, TQ_SLOT( saveConfig() ) );
}

void KPrWebPresentationCreateDialog::resizeEvent( TQResizeEvent *e )
{
    TQDialog::resizeEvent( e );
    back->resize( size() );
}

void KPrWebPresentationCreateDialog::saveConfig()
{
    TQString filename = webPres.getConfig();
    if ( TQFileInfo( filename ).exists() )
        filename = TQFileInfo( filename ).absFilePath();
    else
        filename = TQString();

    KFileDialog fd (filename, i18n("*.kpweb|KPresenter Web-Presentation (*.kpweb)"),
                    0/*parent*/, 0/*name*/, true/*modal*/);
    fd.setCaption (i18n ("Save Web Presentation Configuration"));
    fd.setOperationMode (KFileDialog::Saving);
    fd.setMode (KFile::File | KFile::LocalOnly);

    if (fd.exec ())
    {
        webPres.setConfig( fd.selectedFile () );
        webPres.saveConfig();
    }
}

#include "KPrGradient.h"
#include "KPrWebPresentation.moc"
