/*
Copyright (c) 2000 Simon Hausmann <hausmann@kde.org>
Copyright (c) 2000 Lars Knoll <knoll@kde.org>
Copyright (c) 1999 Preston Brown <pbrown@kde.org>
Copyright (c) 1999, 2000 Matthias Ettrich <ettrich@kde.org>

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include "dcopobject.h"
#include "global.h"
#include "dcopc.h"

#include <string.h>
#include <stdlib.h>

#define OBJECT_CLASS(obj) DCOP_OBJECT_CLASS(GTK_OBJECT_GET_CLASS(obj))

typedef struct _DcopObjectPrivate DcopObjectPrivate;

struct _DcopObjectPrivate
{
    gchar *id;
};

/* initialization */
static void dcop_object_class_init(DcopObjectClass *klass);

static void dcop_object_init(DcopObject *part);

/* GtkObject methods */
static void dcop_object_destroy(GtkObject *object);

static GtkObjectClass *parent_class = 0;

GHashTable *object_dict = 0;

GtkType dcop_object_get_type(void)
{
  static GtkType dcop_object_type = 0;
  if (!dcop_object_type)
  {
    static const GtkTypeInfo dcop_object_info =
    {
      (gchar *)"DcopObject",
      sizeof(DcopObject),
      sizeof(DcopObjectClass),
      (GtkClassInitFunc)dcop_object_class_init,
      (GtkObjectInitFunc)dcop_object_init,
      0,
      0,
      0
    };
    dcop_object_type = gtk_type_unique(GTK_TYPE_OBJECT, &dcop_object_info);
  }
  return dcop_object_type;
}

gboolean dcop_object_real_process( DcopObject *obj, const gchar *fun, dcop_data *data,
                                   gchar **reply_type, dcop_data **reply_data );

gboolean dcop_object_real_process_dynamic( DcopObject *obj, const gchar *fun, dcop_data *data,
                                           gchar **reply_type, dcop_data **reply_data );

GList *dcop_object_real_functions( DcopObject *obj );

GList *dcop_object_real_interfaces( DcopObject *obj );

void dcop_object_class_init(DcopObjectClass *klass)
{
    GtkObjectClass     *object_class;
    object_class    = (GtkObjectClass *)klass;

    parent_class = (GtkObjectClass *)gtk_type_class(gtk_object_get_type());

    object_class->destroy = dcop_object_destroy;

    klass->process = dcop_object_real_process;
    klass->process_dynamic = dcop_object_real_process_dynamic;
    klass->functions = dcop_object_real_functions;
    klass->interfaces = dcop_object_real_interfaces;

    g_message( "dcop_object_class_init(DcopObjectClass *klass)\n");
}

void dcop_object_init(DcopObject *obj)
{
    DcopObjectPrivate *d;
    /* eheh :-) (Simon)*/
/*    d = new DcopObjectPrivate();*/
    d = g_new( DcopObjectPrivate, 1 );
    d->id = 0;
    obj->data = d;

    /* register a unique id*/
    {
	gchar n[1024];
	g_snprintf( n, sizeof( n ), "%p", obj );
	dcop_object_set_id( obj, n ); /* also registers the object at the object_dict*/
    }

    g_message( "dcop_object_init(DcopObject *obj)\n");
}

DcopObject *dcop_object_new (void)
{
    DcopObject *obj = DCOP_OBJECT( gtk_type_new(dcop_object_get_type()) );

    return obj;
}

void dcop_object_destroy( GtkObject *obj )
{
    DcopObject *o = DCOP_OBJECT(obj);
    DcopObjectPrivate *d = (DcopObjectPrivate *)o->data;

    g_message( "dcop_object_destructor %s\n", DCOP_ID(o) );

    g_assert( object_dict );
    g_assert( d->id );
    
    g_hash_table_remove( object_dict, d->id );

    if ( g_hash_table_size( object_dict ) == 0 )
    {
	g_hash_table_destroy( object_dict );
	object_dict = 0;
    }

    g_free(d->id);
    /* Lars, you hack to much C++ :-))) (Simon)*/
/*    delete d;*/
    g_free( d );

    parent_class->destroy(obj);
}

void dcop_object_set_id( DcopObject *obj, const gchar *id )
{
    DcopObjectPrivate *d = (DcopObjectPrivate *) obj->data;

    if ( !object_dict )
	object_dict = g_hash_table_new( g_str_hash, g_str_equal );

    if ( d->id )
	g_hash_table_remove( object_dict, d->id );

    dcop_string_copy( d->id, id );

    g_assert( d->id );

    g_hash_table_insert( object_dict, d->id, obj );
}

const gchar *dcop_object_get_id( DcopObject *obj)
{
    return ((DcopObjectPrivate *) obj->data)->id;
}

gboolean dcop_object_process( DcopObject *obj, const gchar *fun, dcop_data *data,
                              gchar **reply_type, dcop_data **reply_data )
{
    return OBJECT_CLASS(obj)->process( obj, fun, data, reply_type, reply_data );
}

gboolean dcop_object_process_dynamic( DcopObject *obj, const gchar *fun, dcop_data *data,
                                      gchar **reply_type, dcop_data **reply_data )
{
    return OBJECT_CLASS(obj)->process_dynamic( obj, fun, data, reply_type, reply_data );
}

gboolean dcop_object_real_process( DcopObject *obj, const gchar *fun, dcop_data *data,
                                   gchar **reply_type, dcop_data **reply_data )
{
    GList *strlist = 0;
    DcopObjectClass *klass = OBJECT_CLASS(obj);

    if ( strcmp( fun, "interfaces()" ) == 0 )
    {
        *reply_type = g_strdup( "QCStringList" );
        *reply_data = dcop_data_ref( dcop_data_new() );
        strlist = klass->interfaces( obj );

        dcop_marshal_stringlist( *reply_data, strlist );

        dcop_list_free( strlist );
        return TRUE;
    }
    else if ( strcmp( fun, "functions()" ) == 0 )
    {
        *reply_type = strdup( "QCStringList" );
        *reply_data = dcop_data_ref( dcop_data_new() );
        strlist = klass->functions( obj );

        dcop_marshal_stringlist( *reply_data, strlist );

        dcop_list_free( strlist );
        return TRUE;
    }
    return klass->process_dynamic( obj, fun, data, reply_type, reply_data );
}

gboolean dcop_object_real_process_dynamic( DcopObject *client, const gchar *fun, dcop_data *data,
                                           gchar **reply_type, dcop_data **reply_data )
{
    /* empty default implementation*/
    return FALSE;
}

GList *dcop_object_functions( DcopObject *obj )
{
    return OBJECT_CLASS(obj)->functions( obj );
}

GList *dcop_object_interfaces( DcopObject *obj )
{
    return OBJECT_CLASS(obj)->interfaces( obj );
}

GList *dcop_object_real_functions( DcopObject *client )
{
    GList *res = 0;
    res = g_list_append( res, g_strdup( "QCStringList interfaces()" ) );
    res = g_list_append( res, g_strdup( "QCStringList functions()" ) );
    return res;
}

GList *dcop_object_real_interfaces( DcopObject *client )
{
    GList *res = 0;
    res = g_list_append( res, g_strdup( "DCOPObject" ) );
    return res;
}

DcopObject *dcop_object_lookup( const gchar *name )
{
    DcopObject *res = 0;

    if ( !object_dict || !name )
	return NULL;

    return (DcopObject *)g_hash_table_lookup( object_dict, name );
}

gchar *g_partial_id = 0;
size_t g_id_len = 0;

static void dcop_object_match_internal( gpointer key, gpointer val, gpointer user_data )
{
    GList **lst = (GList **)user_data;
    gchar *nam = (gchar *)key;
    DcopObjectPrivate *d = ((DcopObject *)val)->data;
    
    if ( strncmp( d->id, g_partial_id, g_id_len ) == 0 )
	*lst = g_list_append( *lst, (DcopObject *)val );
}

GList *dcop_object_match( const gchar *partial_id )
{
    GList *res = 0;
    GList *it = 0;
    size_t id_len = strlen( partial_id );

    if ( !object_dict )
	return res;

    g_hash_table_foreach( object_dict, dcop_object_match_internal, &res );

    /*
    if ( !object_list)
        return res;

    it = g_list_first( object_list );

    while ( it )
    {
        DcopObjectPrivate *d = (DcopObjectPrivate *)((DcopObject *)it->data)->data;

        if ( strncmp( d->id, partial_id, id_len ) == 0 )
            res = g_list_append( res, (DcopObject *)it->data );

        it = g_list_next( it );
    }
    */
    return res;
}
