/*
    irccontactmanager.cpp - Manager of IRC Contacts

    Copyright (c) 2003      by Michel Hermier <michel.hermier@wanadoo.fr>

    Kopete    (c) 2003      by the Kopete developers <kopete-devel@kde.org>

    *************************************************************************
    *                                                                       *
    * This program is free software; you can redistribute it and/or modify  *
    * it under the terms of the GNU General Public License as published by  *
    * the Free Software Foundation; either version 2 of the License, or     *
    * (at your option) any later version.                                   *
    *                                                                       *
    *************************************************************************
*/

#include "ircusercontact.h"
#include "ircaccount.h"
#include "irccontactmanager.h"
#include "ircprotocol.h"
#include "ircsignalhandler.h"

#include "ircservercontact.h"
#include "ircchannelcontact.h"

#include "kircengine.h"

#include <kopeteaccountmanager.h>
#include <kopetemetacontact.h>
#include <kopetecontactlist.h>
#include <kopeteview.h>

#include <tdeconfig.h>
#include <tdestandarddirs.h>

#include <tqtimer.h>

IRCContactManager::IRCContactManager(const TQString &nickName, IRCAccount *account, const char *name)
	: TQObject(account, name),
	  m_channels( TQDict<IRCChannelContact>( 17, false ) ),
	  m_users( TQDict<IRCUserContact>( 577, false ) ),
	  m_account( account )
{
	m_mySelf = findUser(nickName);

	Kopete::MetaContact *m = new Kopete::MetaContact();
//	m->setTemporary( true );
	m_myServer = new IRCServerContact(this, account->networkName(), m);

	TQObject::connect(account->engine(), TQ_SIGNAL(incomingMessage(const TQString &, const TQString &, const TQString &)),
			this, TQ_SLOT(slotNewMessage(const TQString &, const TQString &, const TQString &)));

	TQObject::connect(account->engine(), TQ_SIGNAL(incomingPrivMessage(const TQString &, const TQString &, const TQString &)),
			this, TQ_SLOT(slotNewPrivMessage(const TQString &, const TQString &, const TQString &)));

	TQObject::connect(account->engine(), TQ_SIGNAL(incomingNickChange(const TQString &, const TQString &)),
			this, TQ_SLOT( slotNewNickChange(const TQString&, const TQString&)));

	TQObject::connect(account->engine(), TQ_SIGNAL(successfullyChangedNick(const TQString &, const TQString &)),
			this, TQ_SLOT( slotNewNickChange(const TQString &, const TQString &)));

	TQObject::connect(account->engine(), TQ_SIGNAL(incomingUserOnline(const TQString &)),
			this, TQ_SLOT( slotIsonRecieved()));

	TQObject::connect(Kopete::ContactList::self(), TQ_SIGNAL(metaContactAdded( Kopete::MetaContact * )),
			this, TQ_SLOT( slotContactAdded( Kopete::MetaContact* )));

	socketTimeout = 15000;
	TQString timeoutPath = locate( "config", "tdeioslaverc" );
	if( !timeoutPath.isEmpty() )
	{
		TDEConfig config( timeoutPath );
		socketTimeout = config.readNumEntry( "ReadTimeout", 15 ) * 1000;
	}

	m_NotifyTimer = new TQTimer(this);
	TQObject::connect(m_NotifyTimer, TQ_SIGNAL(timeout()),
			this, TQ_SLOT(checkOnlineNotifyList()));
	m_NotifyTimer->start(30000); // check online every 30sec

	new IRCSignalHandler(this);
}

void IRCContactManager::slotNewNickChange(const TQString &oldnick, const TQString &newnick)
{
	IRCUserContact *c =  m_users[ oldnick ];
	if( c )
	{
		m_users.insert(newnick, c);
		m_users.remove(oldnick);
	}
}

void IRCContactManager::slotNewMessage(const TQString &originating, const TQString &channel, const TQString &message)
{
	IRCContact *from = findUser(originating);
	IRCChannelContact *to = findChannel(channel);
	emit privateMessage(from, to, message);
}

void IRCContactManager::slotNewPrivMessage(const TQString &originating, const TQString &user, const TQString &message)
{
	IRCContact *from = findUser(originating);
	IRCUserContact *to = findUser(user);
	emit privateMessage(from, to, message);
}

void IRCContactManager::unregister(Kopete::Contact *contact)
{
	unregisterChannel(contact, true);
	unregisterUser(contact, true);
}

TQValueList<IRCChannelContact*> IRCContactManager::findChannelsByMember( IRCUserContact *contact )
{
	TQValueList<IRCChannelContact*> retVal;
	for( TQDictIterator<IRCChannelContact> it(m_channels); it.current(); ++it )
	{
		if( it.current()->manager(Kopete::Contact::CannotCreate) )
		{
			if( contact == m_mySelf )
				retVal.push_back( it.current() );
			else
			{
				bool c = true;

				Kopete::ContactPtrList members = it.current()->manager()->members();
				for( TQPtrListIterator<Kopete::Contact> it2( members ); c && it2.current(); ++it2 )
				{
					if( it2.current() == contact )
					{
						retVal.push_back( it.current() );
						c = false;
					}
				}
			}
		}
	}

	return retVal;
}

IRCChannelContact *IRCContactManager::findChannel(const TQString &name, Kopete::MetaContact *m)
{
	IRCChannelContact *channel = m_channels[ name ];

	if ( !channel )
	{
		if( !m )
		{
			m = new Kopete::MetaContact();
			m->setTemporary( true );
		}

		channel = new IRCChannelContact(this, name, m);
		m_channels.insert( name, channel );
		TQObject::connect(channel, TQ_SIGNAL(contactDestroyed(Kopete::Contact *)),
			this, TQ_SLOT(unregister(Kopete::Contact *)));
	}

	return channel;
}

IRCChannelContact *IRCContactManager::existChannel( const TQString &channel ) const
{
	return m_channels[ channel ];
}

void IRCContactManager::unregisterChannel(Kopete::Contact *contact, bool force )
{
	IRCChannelContact *channel = (IRCChannelContact*)contact;
	if( force || (
		channel!=0 &&
		!channel->isChatting() &&
		channel->metaContact()->isTemporary() ) )
	{
		m_channels.remove( channel->nickName() );
	}
}

IRCUserContact *IRCContactManager::findUser(const TQString &name, Kopete::MetaContact *m)
{
	IRCUserContact *user = m_users[name.section('!', 0, 0)];

	if ( !user )
	{
		if( !m )
		{
			m = new Kopete::MetaContact();
			m->setTemporary( true );
		}

		user = new IRCUserContact(this, name, m);
		m_users.insert( name, user );
		TQObject::connect(user, TQ_SIGNAL(contactDestroyed(Kopete::Contact *)),
				this, TQ_SLOT(unregister(Kopete::Contact *)));
	}

	return user;
}

IRCUserContact *IRCContactManager::existUser( const TQString &user ) const
{
	return m_users[user];
}

IRCContact *IRCContactManager::findContact( const TQString &id, Kopete::MetaContact *m )
{
	if( KIRC::Entity::isChannel(id) )
		return findChannel( id, m );
	else
		return findUser( id, m );
}

IRCContact *IRCContactManager::existContact( const KIRC::Engine *engine, const TQString &id )
{
	TQDict<Kopete::Account> accounts = Kopete::AccountManager::self()->accounts( IRCProtocol::protocol() );
	TQDictIterator<Kopete::Account> it(accounts);
	for( ; it.current(); ++it )
	{
		IRCAccount *account = (IRCAccount *)it.current();
		if( account && account->engine() == engine )
			return account->contactManager()->existContact(id);
	}
	return 0L;
}

IRCContact *IRCContactManager::existContact( const TQString &id ) const
{
	if( KIRC::Entity::isChannel(id) )
		return existChannel( id );
	else
		return existUser( id );
}

void IRCContactManager::unregisterUser(Kopete::Contact *contact, bool force )
{
	IRCUserContact *user = (IRCUserContact *)contact;
	if( force || (
		user!=0 &&
		user!=mySelf() &&
		!user->isChatting() &&
		user->metaContact()->isTemporary() ) )
	{
		m_users.remove( user->nickName() );
	}
}

void IRCContactManager::slotContactAdded( Kopete::MetaContact *contact )
{
	for( TQPtrListIterator<Kopete::Contact> it( contact->contacts() ); it.current(); ++it )
	{
		if( it.current()->account() == m_account )
		{
			addToNotifyList( static_cast<IRCContact*>( it.current() )->nickName() );
		}
	}
}

void IRCContactManager::addToNotifyList(const TQString &nick)
{
 	if (!m_NotifyList.contains(nick.lower()))
	{
		m_NotifyList.append(nick);
		checkOnlineNotifyList();
	}
}

void IRCContactManager::removeFromNotifyList(const TQString &nick)
{
	if (m_NotifyList.contains(nick.lower()))
		m_NotifyList.remove(nick.lower());
}

void IRCContactManager::checkOnlineNotifyList()
{
	if( m_account->engine()->isConnected() )
	{
		isonRecieved = false;
		m_account->engine()->ison( m_NotifyList );
		//TQTimer::singleShot( socketTimeout, this, TQ_SLOT( slotIsonTimeout() ) );
	}
}

void IRCContactManager::slotIsonRecieved()
{
	isonRecieved = true;
}

void IRCContactManager::slotIsonTimeout()
{
	if( !isonRecieved )
		m_account->engine()->quit("", true);
}

#include "irccontactmanager.moc"
