/*******************************************************************************
**
** Filename   : templatemanagementdialog.cpp
** Created on : 05 June, 2005
** Copyright  : (c) 2005 Till Adam
** Email      : <adam@kde.org>
**
*******************************************************************************/

/*******************************************************************************
**
**   This program is free software; you can redistribute it and/or modify
**   it under the terms of the GNU General Public License as published by
**   the Free Software Foundation; either version 2 of the License, or
**   (at your option) any later version.
**
**   It is distributed in the hope that it will be useful, but
**   WITHOUT ANY WARRANTY; without even the implied warranty of
**   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
**   General Public License for more details.
**
**   You should have received a copy of the GNU General Public License
**   along with this program; if not, write to the Free Software
**   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**   In addition, as a special exception, the copyright holders give
**   permission to link the code of this program with any edition of
**   the TQt library by Trolltech AS, Norway (or with modified versions
**   of TQt that use the same license as TQt), and distribute linked
**   combinations including the two.  You must obey the GNU General
**   Public License in all respects for all of the code used other than
**   TQt.  If you modify this file, you may extend this exception to
**   your version of the file, but you are not obligated to do so.  If
**   you do not wish to do so, delete this exception statement from
**   your version.
**
*******************************************************************************/
#include "templatemanagementdialog.h"

#include <tqstringlist.h>
#include <tqtimer.h>

#include <kpushbutton.h>
#include <kinputdialog.h>
#include <tdelocale.h>
#include <tdemessagebox.h>

TemplateManagementDialog::TemplateManagementDialog(TQWidget *parent, const TQStringList &templates )
    :KDialogBase( parent, "template_management_dialog", true,
                        i18n("Manage Templates"), Ok|Cancel, Ok, true , i18n("Apply Template")),
      m_templates( templates ), m_newTemplate( TQString() ), m_changed( false )
{
  m_base = new TemplateManagementDialog_base( this, "template_management_dialog_base" );
  setMainWidget( m_base );
  connect( m_base->m_buttonAdd, TQ_SIGNAL( clicked() ),
           TQ_SLOT( slotAddTemplate() ) );
  connect( m_base->m_buttonDelete, TQ_SIGNAL( clicked() ),
           TQ_SLOT( slotDeleteTemplate() ) );
  m_base->m_listBox->insertStringList( m_templates );
  connect( m_base->m_listBox, TQ_SIGNAL( selectionChanged( TQListBoxItem * ) ),
           TQ_SLOT( slotUpdateDeleteButton( TQListBoxItem * ) ) );
  connect( m_base->m_buttonApply, TQ_SIGNAL( clicked() ),
           TQ_SLOT( slotApplyTemplate() ) );

}

void TemplateManagementDialog::slotAddTemplate()
{
  bool ok;
  bool duplicate = false;
  const TQString newTemplate = KInputDialog::getText( i18n("Template Name"),
                                       i18n("Please enter a name for the new template:"),
                                       i18n("New Template"), &ok );
  if ( newTemplate.isEmpty() || !ok ) return;
  if ( m_templates.find( newTemplate) != m_templates.end() ) {
    int rc = KMessageBox::warningContinueCancel( this, i18n("A template with that name already exists, do you want to overwrite it?."), i18n("Duplicate Template Name"), i18n("Overwrite"));
    if ( rc == KMessageBox::Cancel ) {
      TQTimer::singleShot(0, this, TQ_SLOT( slotAddTemplate() ) );
      return;
    }
    duplicate = true;
  }
  if ( !duplicate ) {
    m_templates.append( newTemplate );
    m_base->m_listBox->clear();
    m_base->m_listBox->insertStringList( m_templates );
  }
  m_newTemplate = newTemplate;
  m_changed = true;
  // From this point on we need to keep the original event around until the user has
  // closed the dialog, applying a template would make little sense
  m_base->m_buttonApply->setEnabled( false );
  // neither does adding it again
  m_base->m_buttonAdd->setEnabled( false );
}

void TemplateManagementDialog::slotDeleteTemplate()
{
  TQListBoxItem *const item = m_base->m_listBox->selectedItem();
  if ( !item ) return; // can't happen (TM)
  unsigned int current = m_base->m_listBox->index(item);
  m_templates.remove( item->text() );
  m_base->m_listBox->removeItem( m_base->m_listBox->currentItem() );
  m_changed = true;
  m_base->m_listBox->setSelected(TQMAX(current -1, 0), true);
}

void TemplateManagementDialog::slotUpdateDeleteButton( TQListBoxItem *item )
{
  m_base->m_buttonDelete->setEnabled( item != 0 );
}

void TemplateManagementDialog::slotApplyTemplate()
{
  // Once the user has applied the current template to the event, it makes no sense to add it again
  m_base->m_buttonAdd->setEnabled( false );
  const TQString &cur = m_base->m_listBox->currentText();
  if ( !cur.isEmpty() && cur != m_newTemplate )
    emit loadTemplate( cur );
}

void TemplateManagementDialog::slotOk()
{
  // failure is not an option *cough*
  if ( !m_newTemplate.isEmpty() )
    emit saveTemplate( m_newTemplate );
  if ( m_changed )
    emit templatesChanged( m_templates );
  KDialogBase::slotOk();
}


#include "templatemanagementdialog.moc"
