/***************************************************************************
 *   Copyright (C) 2006 by Peter Penz                                      *
 *   peter.penz@gmx.at                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.             *
 ***************************************************************************/

#include "dolphinstatusbar.h"
#include <kprogress.h>
#include <tqlabel.h>
#include <tqtimer.h>
#include <kiconloader.h>

#include "dolphinview.h"
#include "statusbarmessagelabel.h"
#include "statusbarspaceinfo.h"

DolphinStatusBar::DolphinStatusBar(DolphinView* parent) :
    TQHBox(parent),
    m_messageLabel(0),
    m_spaceInfo(0),
    m_progressBar(0),
    m_progress(100)
{
    m_messageLabel = new StatusBarMessageLabel(this);
    m_messageLabel->setSizePolicy(TQSizePolicy::Ignored, TQSizePolicy::Fixed);

    m_spaceInfo = new StatusBarSpaceInfo(this);
    m_spaceInfo->setURL(parent->url());

    m_progressText = new TQLabel(this);
    m_progressText->hide();

    m_progressBar = new KProgress(this);
    m_progressBar->hide();

    m_progressTimer = new TQTimer(this);
    connect(m_progressTimer, TQ_SIGNAL(timeout()),
            this, TQ_SLOT(slotProgressTimer()));

    const TQSize size(m_progressBar->sizeHint());
    m_progressBar->setMaximumWidth(size.width());
    setMinimumHeight(size.height());
    m_messageLabel->setMinimumTextHeight(size.height());

    connect(parent, TQ_SIGNAL(signalURLChanged(const KURL&)),
            this, TQ_SLOT(slotURLChanged(const KURL&)));
}


DolphinStatusBar::~DolphinStatusBar()
{
}

void DolphinStatusBar::setMessage(const TQString& msg,
                                  Type type)
{
    m_messageLabel->setText(msg);
    if (msg.isEmpty() || (msg == m_defaultText)) {
        type = Default;
    }
    m_messageLabel->setType(type);

    if ((type == Error) && (m_progress < 100)) {
        // If an error message is shown during a progress is ongoing,
        // the (never finishing) progress information should be hidden immediately
        // (invoking 'setProgress(100)' only leads to a delayed hiding).
        m_progressBar->hide();
        m_progressText->hide();
        setProgress(100);
    }
}

DolphinStatusBar::Type DolphinStatusBar::type() const
{
    return m_messageLabel->type();
}

TQString DolphinStatusBar::message() const
{
    return m_messageLabel->text();
}

void DolphinStatusBar::setProgressText(const TQString& text)
{
    m_progressText->setText(text);
}

TQString DolphinStatusBar::progressText() const
{
    return m_progressText->text();
}

void DolphinStatusBar::setProgress(int percent)
{
    if (percent < 0) {
        percent = 0;
    }
    else if (percent > 100) {
        percent = 100;
    }

    m_progress = percent;
    m_progressBar->setProgress(m_progress);
    m_progressTimer->start(300, true);

    const TQString msg(m_messageLabel->text());
    if (msg.isEmpty() || (msg == m_defaultText)) {
        if (percent == 0) {
            m_messageLabel->setText(TQString());
            m_messageLabel->setType(Default);
        }
        else if (percent == 100) {
            m_messageLabel->setText(m_defaultText);
        }
    }
}

void DolphinStatusBar::clear()
{
    // TODO: check for timeout, so that it's prevented that
    // a message is cleared too early.
    m_messageLabel->setText(m_defaultText);
    m_messageLabel->setType(Default);
}

void DolphinStatusBar::setDefaultText(const TQString& text)
{
    m_defaultText = text;
}

void DolphinStatusBar::slotProgressTimer()
{
    if (m_progress < 100) {
        // progress should be shown
        m_progressBar->show();
        m_progressText->show();
        m_spaceInfo->hide();
    }
    else {
        // progress should not be shown anymore
        m_progressBar->hide();
        m_progressText->hide();
        m_spaceInfo->show();
    }
}

void DolphinStatusBar::slotURLChanged(const KURL& url)
{
    m_spaceInfo->setURL(url);
}

#include "dolphinstatusbar.moc"
