/*
   KD Tools - a set of useful widgets for TQt
   $Id: KDStream.cpp 445690 2005-08-11 17:01:49Z toma $
*/

/****************************************************************************
** Copyright (C) 2001-2005 Klar�lvdalens Datakonsult AB.  All rights reserved.
**
** This file is part of the KD Tools library.
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.
**
** Licensees holding valid commercial KD Tools licenses may use this file in
** accordance with the KD Tools Commercial License Agreement provided with
** the Software.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
** Contact info@klaralvdalens-datakonsult.se if any conditions of this
** licensing are not clear to you.
**
**********************************************************************/

#if defined KDAB_EVAL
#include "../evaldialog/evaldialog.h"
#endif

#include "KDStream.h"
#include <tqcolor.h>
#include <tqpalette.h>
#include <tqcursor.h>
#include <tqdatetime.h>
#include <tqfont.h>
#include <tqpen.h>
#include <tqpoint.h>
#include <tqsize.h>
#include <tqrect.h>
#include <tqregion.h>
#include <tqobject.h>
#include <tqstringlist.h>
#include <tqmetaobject.h>
#include <tqvariant.h>
#include <tqpointarray.h>
#include <tqsizepolicy.h>
#include <tqbitarray.h>
#include <tqstrlist.h>

#include <tqkeysequence.h>
#include <tqpixmap.h>
#include <tqimage.h>

/**
   \file KDStream.cpp
   \class KDStream KDStream.h
   \brief Streaming operators for TQt classes.

   When debugging TQt programs the streaming operators in this class offers
   facilities for printing out values of a number of TQt classes.

   Example:
   \code
   TQPoint point(10,20);
   TQString string("A test");
   TQFont font = tqApp->font();
   KDStream() << "the point is " << point << ", the string is " << string << ", the font is " << font << endl;
   \endcode
*/


/*!
  Creates a KDStream object.
*/
KDStream::KDStream( TQString* outputString ) :_out(outputString)
{
#if defined KDAB_EVAL
    EvalDialog::checkEvalLicense( "KD Tools" );
#endif

}


/*!
  Flushes the data to the stream and destroys the KDStream object.
*/
KDStream::~KDStream()
{
  flush();
}


/*!
  Flushes buffered data to the stream.
*/
void KDStream::flush()
{
  if ( _output.isEmpty() )
    return;

  if ( _out )
      *_out += _output;
  else
      tqDebug( "%s", _output.local8Bit().data() );
  _output = TQString();
}


/*!
  Writes a boolean value to the stream. The value will be represented
  as either "true" or "false".
*/
KDStream& KDStream::operator<<( bool b )
{
  _output += ( b ? TQString::fromLatin1("true") : TQString::fromLatin1("false") );
  return *this;
}

/*!
  Writes a character value to the stream.
*/
KDStream& KDStream::operator<<( char ch )
{
  _output += TQString::fromLatin1("%1").arg( ch );
  return *this;
}


/*!
  Writes a floating point value to the stream.
*/
KDStream& KDStream::operator<<( float num )
{
  _output += TQString::number( num );
  return *this;
}


/*!
  Writes a double-precision floating point value to the stream.
*/
KDStream& KDStream::operator<<( double num )
{
  _output += TQString::number( num );
  return *this;
}


/*!
  Writes a short value to the stream.
*/
KDStream& KDStream::operator<<( short num )
{
  _output += TQString::number( num );
  return *this;
}

/*!
  Writes an unsigned short value to the stream.
*/
KDStream& KDStream::operator<<( unsigned short num )
{
  _output += TQString::number( num );
  return *this;
}

/*!
  Writes an int value to the stream.
*/
KDStream& KDStream::operator<<( int num )
{
  _output += TQString::number( num );
  return *this;
}

/*!
  Writes an unsigned int value to the stream.
*/
KDStream& KDStream::operator<<( unsigned int num )
{
  _output += TQString::number( num );
  return *this;
}

/*!
  Writes a long value to the stream.
*/
KDStream& KDStream::operator<<( long num )
{
  _output += TQString::number( num );
  return *this;
}

/*!
  Writes an unsigned long value to the stream.
*/
KDStream& KDStream::operator<<( unsigned long num )
{
  _output += TQString::number( num );
  return *this;
}

/*!
  Writes a C-style string to the stream.
*/
KDStream& KDStream::operator<<( const char* ch )
{
  *this << TQString(TQString::fromLocal8Bit( ch ));
  return *this;
}

/*!
  Writes a pointer to the stream. The format is platform-dependent and
  defined by fprintf( ..., "%p" ).
*/
KDStream& KDStream::operator<<( const void* p)
{
  _output += TQString().sprintf("%p",p);
  return *this;
}


/*!
  Writes a TQString value to the stream.
*/
KDStream& KDStream::operator<<( const TQString& str )
{
  int index = str.findRev( '\n' );
  if ( index == -1 )
    _output += str;
  else {
    _output += str.left( index )  + '\n';
    flush();
    _output += str.mid( index+1 );
  }
  return *this;
}

/*!
  Writes a TQCString value to the stream.
*/
KDStream& KDStream::operator<<( const TQCString& str )
{
  *this << TQString( str );
  return *this;
}


/*!
  Runs a stream-processing function on the stream.
*/
KDStream& KDStream::operator<<( KDSTREAMFUNC func )
{
  return (*func)(*this);
}

/*!
  Writes a line terminator to the stream.
*/
KDStream& endl( KDStream& stream)
{
  stream << TQString::fromLatin1("\n");
  stream.flush();
  return stream;
}

/*!
  Flushes the output buffers.
*/
KDStream& flush( KDStream& stream)
{
  stream.flush();
  return stream;
}

/*!
  Writes a TQChar value to the stream.
*/
KDStream& KDStream::operator<<( const TQChar& ch )
{
  _output += TQString( ch );
  return *this;
}


/*!
  Writes a TQColor value to the stream. See \ref TQColor2Str for a
  description of the output format.
*/
KDStream& KDStream::operator<<( const TQColor& col )
{
  _output += TQColor2Str( col );
  return *this;
}

/*!
  Writes a TQColorGroup value to the stream. Each color role output
  with its name and the corresponding color value.
*/
KDStream& KDStream::operator<<( const TQColorGroup& colgrp )
{
  _output +=
    TQString::fromLatin1("foreground: ") + TQColor2Str(colgrp.foreground()) + TQString::fromLatin1(", ") +
    TQString::fromLatin1("button: ") + TQColor2Str(colgrp.button()) + TQString::fromLatin1(", ") +
    TQString::fromLatin1("light: ") + TQColor2Str(colgrp.light()) + TQString::fromLatin1(", ") +
    TQString::fromLatin1("dark: ") + TQColor2Str(colgrp.dark()) + TQString::fromLatin1(", ") +
    TQString::fromLatin1("mid: ") + TQColor2Str(colgrp.mid()) + TQString::fromLatin1(", ") +
    TQString::fromLatin1("text: ") + TQColor2Str(colgrp.text()) + TQString::fromLatin1(", ") +
    TQString::fromLatin1("base: ") + TQColor2Str(colgrp.base()) + TQString::fromLatin1(", ") +
    TQString::fromLatin1("background: ") + TQColor2Str(colgrp.background()) + TQString::fromLatin1(", ") +
    TQString::fromLatin1("midlight: ") + TQColor2Str(colgrp.midlight()) + TQString::fromLatin1(", ") +
    TQString::fromLatin1("brightText: ") + TQColor2Str(colgrp.brightText()) + TQString::fromLatin1(", ") +
    TQString::fromLatin1("buttonText: ") + TQColor2Str(colgrp.buttonText()) + TQString::fromLatin1(", ") +
    TQString::fromLatin1("shadow: ") + TQColor2Str(colgrp.shadow()) + TQString::fromLatin1(", ") +
    TQString::fromLatin1("highlight: ") + TQColor2Str(colgrp.highlight()) + TQString::fromLatin1(", ") +
    TQString::fromLatin1("highlightedText: ") + TQColor2Str(colgrp.highlightedText());
  return *this;
}

/*!
  Writes a TQPalette value to the stream. Each color group is output
  with its role and the corresponding TQColorGroup value.
*/
KDStream& KDStream::operator<<( const TQPalette& palette )
{
  *this << TQString::fromLatin1("active: ") << palette.active() << endl
        << TQString::fromLatin1("inactive: ") << palette.inactive() << endl
        << TQString::fromLatin1("diabled: ") << palette.disabled();

  return *this;
}

/*!
  Writes a TQCursor value to the stream. Each cursor is output with its
  name as listed in the \a TQCursor reference documentation.
*/
KDStream& KDStream::operator<<( const TQCursor& cursor )
{

  TQString type;
  switch ( cursor.shape() ) {
  case TQt::ArrowCursor: type = TQString::fromLatin1("ArrowCursor"); break;
  case TQt::UpArrowCursor: type = TQString::fromLatin1("UpArrowCursor"); break;
  case TQt::CrossCursor: type = TQString::fromLatin1("CrossCursor"); break;
  case TQt::WaitCursor: type = TQString::fromLatin1("WaitCursor"); break;
  case TQt::IbeamCursor: type = TQString::fromLatin1("IbeamCursor"); break;
  case TQt::SizeVerCursor: type = TQString::fromLatin1("SizeVerCursor"); break;
  case TQt::SizeHorCursor: type = TQString::fromLatin1("SizeHorCursor"); break;
  case TQt::SizeBDiagCursor: type = TQString::fromLatin1("SizeBDiagCursor"); break;
  case TQt::SizeFDiagCursor: type = TQString::fromLatin1("SizeFDiagCursor"); break;
  case TQt::SizeAllCursor: type = TQString::fromLatin1("SizeAllCursor"); break;
  case TQt::BlankCursor: type = TQString::fromLatin1("BlankCursor"); break;
  case TQt::SplitVCursor: type = TQString::fromLatin1("SplitVCursor"); break;
  case TQt::SplitHCursor: type = TQString::fromLatin1("SplitHCursor"); break;
  case TQt::PointingHandCursor: type = TQString::fromLatin1("PointingHandCursor"); break;
  case TQt::ForbiddenCursor: type = TQString::fromLatin1("ForbiddenCursor"); break;
  case TQt::BitmapCursor: type = TQString::fromLatin1("BitmapCursor"); break;
  }

  _output += type;

  return *this;
}

/*!
  Writes a TQDate value to the stream. The format is the one defined by
  TQDate::toString() and may be system-dependent.
*/
KDStream& KDStream::operator<<( const TQDate& date )
{
  _output += date.toString();
  return *this;
}

/*!
  Writes a TQDateTime value to the stream. The format is the one
  defined by TQDateTime::toString() and may be system-dependent.
*/
KDStream& KDStream::operator<<( const TQDateTime& datetime )
{
  _output += datetime.toString();
  return *this;
}

/*!
  Writes a TQTime value to the stream. The format is the one defined by
  TQTime::toString() and may be system-dependent.
*/
KDStream& KDStream::operator<<( const TQTime& time )
{
  _output += time.toString();
  return *this;
}

/*!
  Writes a the raw name of a TQFont value to the stream.
*/
KDStream& KDStream::operator<<( const TQFont& font )
{
  _output += font.rawName();
  return *this;
}

/*!
  Writes a TQPen value to the stream. The format is "TQPen" plus the
  width, the color, and the style as defined in the \a TQPen reference
  documentation.
*/
KDStream& KDStream::operator<<( const TQPen& pen )
{
  TQString style;
  switch ( pen.style() ) {
  case TQt::NoPen: style = TQString::fromLatin1("NoPen"); break;
  case TQt::SolidLine: style = TQString::fromLatin1("SolidLine"); break;
  case TQt::DashLine: style = TQString::fromLatin1("DashLine"); break;
  case TQt::DotLine: style = TQString::fromLatin1("DotLine"); break;
  case TQt::DashDotLine: style = TQString::fromLatin1("DashDotLine"); break;
  case TQt::DashDotDotLine : style = TQString::fromLatin1("DashDotDotLine "); break;
  case TQt::MPenStyle : break; // ignore
  }

  _output += TQString::fromLatin1("TQPen(%1,%2,%3)")
    .arg( pen.width() )
    .arg( TQColor2Str( pen.color() ) )
    .arg( style );
  return *this;
}

/*!
  Writes a TQPoint value to the stream. The format is "(x,y)".
*/
KDStream& KDStream::operator<<( const TQPoint& point )
{
  _output += TQString::fromLatin1("(%1,%2)").arg(point.x()).arg(point.y() );
  return *this;
}

/*!
  Writes a TQSize value to the stream. The format is "(w x h)".
*/
KDStream& KDStream::operator<<( const TQSize& size )
{
  _output += TQString::fromLatin1("%1x%2").arg(size.width()).arg(size.height());
  return *this;
}

/*!
  Writes a TQRect value to the stream. The format is "(width x height
  xoffset xpos yoffset ypos)".
*/
KDStream& KDStream::operator<<( const TQRect& rect )
{
  TQString xplus = (rect.x() >= 0) ? TQString::fromLatin1("+") : TQString::fromLatin1("");
  TQString yplus = (rect.y() >= 0) ? TQString::fromLatin1("+") : TQString::fromLatin1("");
  _output += TQString::fromLatin1("%1x%2%3%4%5%6")
    .arg( rect.width() )
    .arg( rect.height() )
    .arg( xplus )
    .arg( rect.x() )
    .arg( yplus )
    .arg( rect.y() );

  return *this;
}

/*!
  This is a helper method that converts a TQColor object into a
  string.  For the predefined TQt colors, their name is output, for all
  other colors, the output is in the form #RRGGBB (as defined by
  TQColor::name()).
*/
TQString KDStream::TQColor2Str( const TQColor& col )
{
  if ( col == TQt::black )
    return TQString::fromLatin1("black");
  else if ( col == TQt::white )
    return TQString::fromLatin1("white");
  else if ( col == TQt::darkGray )
    return TQString::fromLatin1("darkGray");
  else if ( col == TQt::gray )
    return TQString::fromLatin1("gray");
  else if ( col == TQt::lightGray )
    return TQString::fromLatin1("lightGray");
  else if ( col == TQt::red )
    return TQString::fromLatin1("red");
  else if ( col == TQt::green )
    return TQString::fromLatin1("green");
  else if ( col == TQt::blue )
    return TQString::fromLatin1("blue");
  else if ( col == TQt::cyan )
    return TQString::fromLatin1("cyan");
  else if ( col == TQt::magenta )
    return TQString::fromLatin1("magenta");
  else if ( col == TQt::yellow )
    return TQString::fromLatin1("yellow");
  else if ( col == TQt::darkRed )
    return TQString::fromLatin1("darkRed");
  else if ( col == TQt::darkGreen )
    return TQString::fromLatin1("darkGreen");
  else if ( col == TQt::darkBlue )
    return TQString::fromLatin1("darkBlue");
  else if ( col == TQt::darkCyan )
    return TQString::fromLatin1("darkCyan");
  else if ( col == TQt::darkMagenta )
    return TQString::fromLatin1("darkMagenta");
  else if ( col == TQt::darkYellow )
    return TQString::fromLatin1("darkYellow");
  else if ( col == TQt::color0 )
    return TQString::fromLatin1("color0");
  else if ( col == TQt::color1 )
    return TQString::fromLatin1("color1");
  else
    return col.name();
}

/*!
  Writes a TQObject value to the stream. Included information is the
  class name, the object name, the properties and their types.
*/
KDStream& KDStream::operator<<( const TQObject& obj )
{
  *this << TQString::fromLatin1(obj.className()) + TQString::fromLatin1("(") + TQString::fromLatin1(obj.name()) << TQString::fromLatin1("):")<< endl;
  TQMetaObject* meta = obj.metaObject();
  TQStrList props = meta->propertyNames(true);

  unsigned int maxWidth = 0;
  for ( TQStrListIterator it(props) ; *it; ++it ) {
    maxWidth = TQMAX( maxWidth, TQString::fromLatin1(*it).length() );
  }

  for ( TQStrListIterator it2(props) ; *it2; ++it2 ) {
    *this << TQString::fromLatin1("  ") << TQString::fromLatin1(*it2).leftJustify(maxWidth) << TQString::fromLatin1(": [") << obj.property(*it2) << TQString::fromLatin1("]") << endl;
  }
  return *this;
}

/*!
  Writes a TQVariant value to the stream. The format is dependent on
  the actual contents of the TQVariant object.
*/
KDStream& KDStream::operator<<( const TQVariant& var)
{
  switch (var.type() )
    {
    case TQVariant::Invalid: *this << TQString::fromLatin1("*INVALID*"); break;
    case TQVariant::Map: *this << var.toMap(); break;
    case TQVariant::List: *this << var.toList(); break;
    case TQVariant::String: *this << var.toString(); break;
    case TQVariant::StringList: *this << var.toStringList(); break;
    case TQVariant::Font: *this << var.toFont(); break;
    case TQVariant::Pixmap: *this << var.toPixmap();break;

    case TQVariant::Brush: *this << var.toBrush(); break;
    case TQVariant::Rect: *this << var.toRect(); break;
    case TQVariant::Size: *this << var.toSize(); break;
    case TQVariant::Color: *this << var.toColor(); break;
    case TQVariant::Palette: *this << var.toPalette(); break;
    case TQVariant::ColorGroup: *this << var.toColorGroup(); break;
    case TQVariant::IconSet: *this << TQString::fromLatin1("-"); break;
    case TQVariant::Point: *this << var.toPoint(); break;
    case TQVariant::Image: *this << var.toImage(); break;
    case TQVariant::Int: *this << var.toInt(); break;
    case TQVariant::UInt: *this << var.toUInt(); break;
    case TQVariant::Bool: *this << var.toBool(); break;
    case TQVariant::Double: *this << var.toDouble(); break;
    case TQVariant::CString: *this << var.toCString(); break;
    case TQVariant::PointArray: *this << var.toPointArray(); break;
    case TQVariant::Region: *this << TQString::fromLatin1("-"); break;
    case TQVariant::Bitmap: *this << TQString::fromLatin1("-"); break;
    case TQVariant::Cursor: *this << var.toCursor(); break;
    case TQVariant::SizePolicy: *this << var.toSizePolicy(); break;
    case TQVariant::Date: *this << var.toDate(); break;
    case TQVariant::Time: *this << var.toTime(); break;
    case TQVariant::DateTime: *this << var.toDateTime(); break;
    case TQVariant::ByteArray: *this << TQCString(var.toByteArray()); break;
    case TQVariant::BitArray: *this << var.toBitArray(); break;
    case TQVariant::KeySequence: *this << var.toKeySequence(); break;
    case TQVariant::Pen: *this << var.toPen(); break;
    }
  return *this;
}

/*!
  Writes a TQBrush value to the stream. The format is "TQBrush" plus the
  brush style as listed in the \a TQBrush reference documentation and
  the brush color.
*/
KDStream& KDStream::operator<<( const TQBrush& brush)
{
  TQString style;
  switch ( brush.style() )
    {
    case TQt::NoBrush: style = TQString::fromLatin1("NoBrush"); break;
    case TQt::SolidPattern: style = TQString::fromLatin1("SolidPattern"); break;
    case TQt::Dense1Pattern: style = TQString::fromLatin1("Dense1Pattern"); break;
    case TQt::Dense2Pattern: style = TQString::fromLatin1("Dense2Pattern"); break;
    case TQt::Dense3Pattern: style = TQString::fromLatin1("Dense3Pattern"); break;
    case TQt::Dense4Pattern: style = TQString::fromLatin1("Dense4Pattern"); break;
    case TQt::Dense5Pattern: style = TQString::fromLatin1("Dense5Pattern"); break;
    case TQt::Dense6Pattern: style = TQString::fromLatin1("Dense6Pattern"); break;
    case TQt::Dense7Pattern: style = TQString::fromLatin1("Dense7Pattern"); break;
    case TQt::HorPattern: style = TQString::fromLatin1("HorPattern"); break;
    case TQt::VerPattern: style = TQString::fromLatin1("VerPattern"); break;
    case TQt::CrossPattern: style = TQString::fromLatin1("CrossPattern"); break;
    case TQt::BDiagPattern: style = TQString::fromLatin1("BDiagPattern"); break;
    case TQt::FDiagPattern: style = TQString::fromLatin1("FDiagPattern"); break;
    case TQt::DiagCrossPattern: style = TQString::fromLatin1("DiagCrossPattern"); break;
    case TQt::CustomPattern: style = TQString::fromLatin1("CustomPattern"); break;
    }
  _output += TQString::fromLatin1("TQBrush(%1,%2)").arg(style).arg(TQColor2Str(brush.color()));
  return *this;
}

/*!
  Writes a TQSizePolicy value to the stream. The output contains the
  horizontal and vertical size policy and whether the policy has a
  "height for width" setting.
*/
KDStream& KDStream::operator<<( const TQSizePolicy& policy)
{
  TQString hor, ver;

  switch ( policy.horData() )
    {
    case TQSizePolicy::Fixed: hor=TQString::fromLatin1("Fixed"); break;
    case TQSizePolicy::Minimum : hor=TQString::fromLatin1("Minimum "); break;
    case TQSizePolicy::Maximum: hor=TQString::fromLatin1("Maximum"); break;
    case TQSizePolicy::Preferred: hor=TQString::fromLatin1("Preferred"); break;
    case TQSizePolicy::MinimumExpanding: hor=TQString::fromLatin1("MinimumExpanding"); break;
    case TQSizePolicy::Expanding: hor=TQString::fromLatin1("Expanding"); break;
    case TQSizePolicy::Ignored: hor=TQString::fromLatin1("Ignored"); break;
    }
  switch ( policy.verData() )
    {
    case TQSizePolicy::Fixed: ver=TQString::fromLatin1("Fixed"); break;
    case TQSizePolicy::Minimum : ver=TQString::fromLatin1("Minimum "); break;
    case TQSizePolicy::Maximum: ver=TQString::fromLatin1("Maximum"); break;
    case TQSizePolicy::Preferred: ver=TQString::fromLatin1("Preferred"); break;
    case TQSizePolicy::MinimumExpanding: ver=TQString::fromLatin1("MinimumExpanding"); break;
    case TQSizePolicy::Expanding: ver=TQString::fromLatin1("Expanding"); break;
    case TQSizePolicy::Ignored: ver=TQString::fromLatin1("Ignored"); break;
    }
  TQString hforw = policy.hasHeightForWidth() ? TQString::fromLatin1("true") : TQString::fromLatin1("false");
  _output += TQString::fromLatin1("TQSizePolicy(hor=%1,ver=%2, hasHeightForWidth=%3)")
    .arg(hor).arg(ver).arg(hforw);
  return *this;
}

/*!
  Writes a TQKeySequence value to the stream. The output format is the
  string value of the key sequence.
*/
KDStream& KDStream::operator<<( const TQKeySequence& keySeq)
{
  _output += TQString(keySeq);
  return *this;
}

/*!
  Writes a TQStrList value to the stream. The output is the individual
  strings.
*/
KDStream& KDStream::operator<<( const TQStrList& list )
{
  KDStream_ptrListStream( *this, TQStrListIterator( list ), false );
  return *this;
}


KDStream& KDStream::operator<<( const TQPixmap& pixmap )
{
    _output += TQString("TQPixmap[null=%1,width=%2,height=%3,depth=%4,hasMask=%5,hasAlpha=%6]")
               .arg(pixmap.isNull()).arg(pixmap.width()).arg(pixmap.height())
               .arg(pixmap.depth()).arg(pixmap.mask() != 0).arg(pixmap.hasAlpha() );
    return *this;
}


KDStream& KDStream::operator<<( const TQImage& pixmap )
{
    _output += TQString("TQImage[null=%1,width=%2,height=%3,depth=%4,hasAlpha=%5]")
               .arg(pixmap.isNull()).arg(pixmap.width()).arg(pixmap.height())
               .arg(pixmap.depth()).arg(pixmap.hasAlphaBuffer() );
    return *this;
}

/* Classes that do not need to be supported:

TQCollection - abstract class // TQt 2
TQBitArray - TQArray implemented.
TQByteArray - TQAray implemented.
TQPointArray - TQAray implemented.
TQGArray - internal class // TQt 2
TQStringList - It's just a TQValueList.

TQGCache // TQt 2
TQGDict // TQt 2
TQGList // TQt 2
TQGVector // TQt 2
TQGCacheIterator // TQt 2
TQAsciiCacheIterator
TQCacheIterator
TQIntCacheIterator
TQGDictIterator // TQt 2
TQAsciiDictIterator
TQDictIterator
TQIntDictIterator
TQPtrDictIterator
TQStrListIterator
TQGListIterator // TQt 2
TQMapConstIterator
TQMapIterator
TQBitVal
TQValueListConstIterator
TQValueListIterator
TQPtrListIterator// TQt 3
TQPtrCollection// TQt 3
TQSortedList - Depricated // TQt 2

*/

/* TQt classes not yet supported:
TQRegion
TQAccel
TQAccessible // TQt 3
TQAccessibleInterface// TQt 3
TQAccessibleObject// TQt 3
TQAction
TQActionGroup
TQApplication
TQAsyncIO // TQt 2
TQBitmap
TQBoxLayout
TQBuffer
TQButton
TQButtonGroup
TQCDEStyle
TQCanvas
TQCanvasEllipse
TQCanvasItem
TQCanvasItemList// TQt 3
TQCanvasLine
TQCanvasPixmap
TQCanvasPixmapArray
TQCanvasPolygon
TQCanvasPolygonalItem
TQCanvasRectangle
TQCanvasSpline// TQt 3
TQCanvasSprite
TQCanvasText
TQCanvasView
TQCheckBox
TQCheckListItem
TQCheckTableItem// TQt 3
TQChildEvent
TQClipboard
TQCloseEvent
TQColorDialog
TQColorDrag
TQComboBox
TQComboTableItem// TQt 3
TQCommonStyle
TQComponentFactory// TQt 3
TQComponentFactoryInterface// TQt 3
TQComponentInterface// TQt 3
TQComponentServerInterface// TQt 3
TQContextMenuEvent// TQt 3
TQCopChannel
TQCustomEvent
TQCustomMenuItem
TQDataBrowser// TQt 3
TQDataPump // TQt 2
TQDataSink // TQt 2
TQDataSource // TQt 2
TQDataStream
TQDataTable// TQt 3
TQDataView// TQt 3
TQDateEdit// TQt 3
TQDateTimeEdit// TQt 3
TQDesktopWidget// TQt 3
TQDial
TQDialog
TQDir
TQDns
TQDockArea// TQt 3
TQDockWindow// TQt 3
TQDomAttr
TQDomCDATASection
TQDomCharacterData
TQDomComment
TQDomDocument
TQDomDocumentFragment
TQDomDocumentType
TQDomElement
TQDomEntity
TQDomEntityReference
TQDomImplementation
TQDomNamedNodeMap
TQDomNode
TQDomNodeList
TQDomNotation
TQDomProcessingInstruction
TQDomText
TQDoubleValidator
TQDragEnterEvent
TQDragLeaveEvent
TQDragMoveEvent
TQDragObject
TQDropEvent
TQDropSite // TQt 2
TQEditorFactory// TQt 3
TQErrorMessage// TQt 3
TQEucJpCodec // TQt 2
TQEucKrCodec // TQt 2
TQEvent
TQFeatureListInterface// TQt 3
TQFile
TQFileDialog
TQFileIconProvider
TQFileInfo
TQFilePreview
TQFocusData
TQFocusEvent
TQFontDatabase
TQFontDialog
TQFontInfo
TQFontManager// TQt 3
TQFontMetrics
TQFrame
TQFtp
TQGL
TQGLColormap// TQt 3
TQGLContext
TQGLFormat
TQGLWidget
TQGLayoutIterator
TQGbkCodec // TQt 2
TQGrid
TQGridLayout
TQGridView// TQt 3
TQGroupBox
TQGuardedPtr
TQHBox
TQHBoxLayout
TQHButtonGroup
TQHGroupBox
TQHeader
TQHideEvent
TQHostAddress
TQHttp// TQt 3
TQIMEvent// TQt 3
TQIODevice
TQIODeviceSource // TQt 2
TQIconDrag
TQIconDragItem
TQIconSet
TQIconView
TQIconViewItem
TQImageConsumer
TQImageDecoder
TQImageDrag
TQImageFormat
TQImageFormatType
TQImageIO
TQInputDialog
TQIntValidator
TQInterlaceStyle
TQJisCodec // TQt 2
TQJpUnicodeConv // TQt 2
TQKeyEvent
TQLCDNumber
TQLNode // TQt 2
TQLabel
TQLayout
TQLayoutItem
TQLayoutIterator
TQLibrary// TQt 3
TQLibraryInterface// TQt 3
TQLineEdit
TQListBox
TQListBoxItem
TQListBoxPixmap
TQListBoxText
TQListView
TQListViewItem
TQListViewItemIterator
TQLocalFs
TQLock// TQt 3
TQMainWindow
TQMap
TQMemArray// TQt 3
TQMenuBar
TQMenuData
TQMessageBox
TQMetaObject
TQMetaProperty
TQMimeSource
TQMimeSourceFactory
TQMotifPlusStyle
TQMotifStyle
TQMouseEvent
TQMoveEvent
TQMovie
TQMultiLineEdit // TQt 2
TQMutex
TQNPInstance
TQNPStream
TQNPWidget
TQNPlugin
TQNetworkOperation
TQNetworkProtocol
TQObject
TQPNGImagePacker
TQPaintDevice
TQPaintDeviceMetrics
TQPaintEvent
TQPainter
TQPicture
TQPixmapCache
TQPlatinumStyle
TQPluginManager// TQt 3
TQPopupMenu
TQPrinter
TQProcess// TQt 3
TQProgressBar
TQProgressDialog
TQPushButton
TQRadioButton
TQRangeControl
TQRegExp
TQRegExpValidator// TQt 3
TQResizeEvent
TQSGIStyle
TQScreen// TQt 3
TQScreenCursor // TQt 2
TQScrollBar
TQScrollView
TQSemaphore
TQSemiModal // TQt 2
TQServerSocket
TQSessionManager
TQSettings// TQt 3
TQShared // TQt 2
TQShowEvent
TQSignal
TQSignalMapper
TQSimpleRichText
TQSizeGrip
TQSjisCodec // TQt 2
TQSlider
TQSocket
TQSocketDevice
TQSocketNotifier
TQSound
TQSpacerItem
TQSpinBox
TQSplitter
TQSql// TQt 3
TQSqlCursor// TQt 3
TQSqlDatabase// TQt 3
TQSqlDriver// TQt 3
TQSqlEditorFactory// TQt 3
TQSqlError// TQt 3
TQSqlField// TQt 3
TQSqlForm// TQt 3
TQSqlIndex// TQt 3
TQSqlPropertyMap// TQt 3
TQSqlQuery// TQt 3
TQSqlRecord// TQt 3
TQSqlResult// TQt 3
TQStatusBar
TQStoredDrag
TQStyle
TQStyleSheet
TQStyleSheetItem
TQTab
TQTabBar
TQTabDialog
TQTabWidget
TQTable
TQTableItem
TQTableSelection
TQTableView // TQt 2
TQTabletEvent// TQt 3
TQTextBrowser
TQTextCodec
TQTextDecoder
TQTextDrag
TQTextEdit// TQt 3
TQTextEncoder
TQTextIStream
TQTextOStream
TQTextStream
TQTextView // TQt 2
TQThread
TQTimeEdit// TQt 3
TQTimer
TQTimerEvent
TQToolBar
TQToolButton
TQToolTip
TQToolTipGroup
TQTranslator
TQTranslatorMessage
TQTsciiCodec // TQt 2
TQUnknownInterface// TQt 3
TQUriDrag
TQUrl
TQUrlInfo// TQt 3
TQUrlOperator
TQUuid// TQt 3
TQVBox
TQVBoxLayout
TQVButtonGroup
TQVGroupBox
TQValidator
TQVariant
TQWMatrix
TQWSDecoration
TQWSKeyboardHandler
TQWSMouseHandler
TQWSServer
TQWSWindow
TQWaitCondition
TQWhatsThis
TQWheelEvent
TQWidget
TQWidgetFactory// TQt 3
TQWidgetItem
TQWidgetStack
TQWindowsMime// TQt 3
TQWindowsStyle
TQWizard
TQWorkspace
TQXmlAttributes
TQXmlContentHandler
TQXmlDTDHandler
TQXmlDeclHandler
TQXmlDefaultHandler
TQXmlEntityResolver
TQXmlErrorHandler
TQXmlInputSource
TQXmlLexicalHandler
TQXmlLocator
TQXmlNamespaceSupport
TQXmlParseException
TQXmlReader
TQXmlSimpleReader
TQXtApplication
TQXtWidget
TQt
*/
