//Author:    Max Howell <max.howell@methylblue.com>, (C) 2003-4
//Copyright: See COPYING file that comes with this distribution

#include "mainWindow.h"
#include "part/part.h"
#include "historyAction.h"

#include <cstdlib>            //std::exit()
#include <tdeaccel.h>           //TDEStdAccel namespace
#include <tdeaction.h>
#include <tdeapplication.h>     //setupActions()
#include <kcombobox.h>        //locationbar
#include <tdeconfig.h>
#include <kdirselectdialog.h> //slotScanDirectory
#include <kedittoolbar.h>     //for editToolbar dialog
#include <kkeydialog.h>
#include <klibloader.h>
#include <tdelocale.h>
#include <tdemessagebox.h>
#include <kshell.h>
#include <kstatusbar.h>
#include <tdetoolbar.h>
#include <kurl.h>
#include <kurlcompletion.h>   //locationbar
#include <tqobjectlist.h>
#include <tqpopupmenu.h>
#include <tqtooltip.h>



namespace Filelight {

MainWindow::MainWindow()
        : KParts::MainWindow()
        , m_part( 0 )
{
    KLibFactory *factory = KLibLoader::self()->factory( "libfilelight" );

    if (!factory) {
       KMessageBox::error( this, i18n("TDE could not find the Filelight Part, or the Filelight Part could not be started. Did you make install?") );
       //exit() seems to not exist inside the std namespace for some users!
       using namespace std;
       exit( 1 ); //don't use TQApplication::exit() - it causes a crash
    }

    m_part = (Part *)factory->create( this, "part", "KParts::ReadOnlyPart" );

    setCentralWidget( m_part->widget() );
    setStandardToolBarMenuEnabled( true );
    setupActions();
    createGUI( m_part );

    stateChanged( "scan_failed" ); //bah! doesn't affect the parts' actions, should I add them to the actionCollection here?

    TQObjectList *buttons = toolBar()->queryList( "TDEToolBarButton" );
    if (buttons->isEmpty())
        KMessageBox::error( this, i18n("Filelight is not installed properly, consequently its menus and toolbars will appear reduced or even empty") );
    delete buttons;

    connect( m_part, TQ_SIGNAL(started( TDEIO::Job* )), TQ_SLOT(scanStarted()) );
    connect( m_part, TQ_SIGNAL(completed()), TQ_SLOT(scanCompleted()) );
    connect( m_part, TQ_SIGNAL(canceled( const TQString& )), TQ_SLOT(scanFailed()) );

    //TODO test these
    connect( m_part, TQ_SIGNAL(canceled( const TQString& )), m_histories, TQ_SLOT(stop()) );
    connect( BrowserExtension::childObject( m_part ), TQ_SIGNAL(openURLNotify()), TQ_SLOT(urlAboutToChange()) );

    TDEConfig* const config = TDEGlobal::config();
    config->setGroup( "general" );
    m_combo->setHistoryItems( config->readPathListEntry( "comboHistory" ) );
    applyMainWindowSettings( config, "window" );
}

inline void
MainWindow::setupActions() //singleton function
{
    TDEActionCollection *const ac = actionCollection();

    m_combo = new KHistoryCombo( this, "history_combo" );
    m_combo->setCompletionObject( new KURLCompletion( KURLCompletion::DirCompletion ) );
    m_combo->setAutoDeleteCompletionObject( true );
    m_combo->setSizePolicy( TQSizePolicy( TQSizePolicy::Expanding, TQSizePolicy::Fixed ) );
    m_combo->setDuplicatesEnabled( false );

    KStdAction::open( this, TQ_SLOT(slotScanDirectory()), ac, "scan_directory" );
    KStdAction::quit( this, TQ_SLOT(close()), ac );
    KStdAction::up( this, TQ_SLOT(slotUp()), ac );
    KStdAction::configureToolbars(this, TQ_SLOT(configToolbars()), ac);
    KStdAction::keyBindings(this, TQ_SLOT(configKeys()), ac);

    new TDEAction( i18n( "Scan &Home Directory" ), "folder_home", CTRL+Key_Home, this, TQ_SLOT(slotScanHomeDirectory()), ac, "scan_home" );
    new TDEAction( i18n( "Scan &Root Directory" ), "folder_red", 0, this, TQ_SLOT(slotScanRootDirectory()), ac, "scan_root" );
    new TDEAction( i18n( "Rescan" ), "reload", TDEStdAccel::reload(), m_part, TQ_SLOT(rescan()), ac, "scan_rescan" );
    new TDEAction( i18n( "Stop" ), "process-stop", TQt::Key_Escape, this, TQ_SLOT(slotAbortScan()), ac, "scan_stop" );
    new TDEAction( i18n( "Clear Location Bar" ), TDEApplication::reverseLayout() ? "clear_left" : "locationbar_erase", 0, m_combo, TQ_SLOT(clearEdit()), ac, "clear_location" );
    new TDEAction( i18n( "Go" ), "key_enter", 0, m_combo, TQ_SIGNAL(returnPressed()), ac, "go" );

    KWidgetAction *combo = new KWidgetAction( m_combo, i18n( "Location Bar" ), 0, 0, 0, ac, "location_bar" );
    m_recentScans = new TDERecentFilesAction( i18n( "&Recent Scans" ), 0, ac, "scan_recent", 8 );
    m_histories = new HistoryCollection( ac, this, "history_collection" );

    ac->action( "scan_directory" )->setText( i18n( "&Scan Directory..." ) );
    m_recentScans->loadEntries( TDEGlobal::config() );
    combo->setAutoSized( true ); //FIXME what does this do?

    connect( m_recentScans, TQ_SIGNAL(urlSelected( const KURL& )), TQ_SLOT(slotScanUrl( const KURL& )) );
    connect( m_combo, TQ_SIGNAL(returnPressed()), TQ_SLOT(slotComboScan()) );
    connect( m_histories, TQ_SIGNAL(activated( const KURL& )), TQ_SLOT(slotScanUrl( const KURL& )) );
}

bool
MainWindow::queryExit()
{
    if( !m_part ) //apparently std::exit() still calls this function, and abort() causes a crash..
       return true;

    TDEConfig* const config = TDEGlobal::config();

    saveMainWindowSettings( config, "window" );
    m_recentScans->saveEntries( config );
    config->setGroup( "general" );
    config->writePathEntry( "comboHistory", m_combo->historyItems() );
    config->sync();

    return true;
}

inline void
MainWindow::configToolbars() //slot
{
    KEditToolbar dialog( factory(), this );
    dialog.showButtonApply( false );

    if( dialog.exec() )
    {
        createGUI( m_part );
        applyMainWindowSettings( kapp->config(), "window" );
    }
}

inline void
MainWindow::configKeys() //slot
{
    KKeyDialog::configure( actionCollection(), this );
}

inline void
MainWindow::slotScanDirectory()
{
    slotScanUrl( KDirSelectDialog::selectDirectory( m_part->url().path(), false, this ) );
}

inline void MainWindow::slotScanHomeDirectory() { slotScanPath( getenv( "HOME" ) ); }
inline void MainWindow::slotScanRootDirectory() { slotScanPath( "/" ); }
inline void MainWindow::slotUp()                { slotScanUrl( m_part->url().upURL() ); }

inline void
MainWindow::slotComboScan()
{
   const TQString path = KShell::tildeExpand(m_combo->lineEdit()->text());
   if (slotScanPath( path ))
      m_combo->addToHistory( path );
}

inline bool
MainWindow::slotScanPath( const TQString &path )
{
   return slotScanUrl( KURL::fromPathOrURL( path ) );
}

bool
MainWindow::slotScanUrl( const KURL &url )
{
   const KURL oldUrl = m_part->url();
   const bool b = m_part->openURL( url );

   if (b) {
      m_histories->push( oldUrl );
      action( "go_back" )->TDEAction::setEnabled( false ); } //FIXME

   return b;
}

inline void
MainWindow::slotAbortScan()
{
    if( m_part->closeURL() ) action( "scan_stop" )->setEnabled( false );
}

inline void
MainWindow::scanStarted()
{
    stateChanged( "scan_started" );
    m_combo->clearFocus();
}

inline void
MainWindow::scanFailed()
{
    stateChanged( "scan_failed" );
    setActionMenuTextOnly( action( "go_up" ), TQString() );
    m_combo->lineEdit()->clear();
}

void
MainWindow::scanCompleted()
{
    TDEAction *goUp  = action( "go_up" );
    const KURL url = m_part->url();

    stateChanged( "scan_complete" );

    m_combo->lineEdit()->setText( m_part->prettyURL() );

    if ( url.path( 1 ) == "/") {
        goUp->setEnabled( false );
        setActionMenuTextOnly( goUp, TQString() );
    }
    else
        setActionMenuTextOnly( goUp, url.upURL().path( 1 ) );

    m_recentScans->addURL( url ); //FIXME doesn't set the tick
}

inline void
MainWindow::urlAboutToChange()
{
   //called when part's URL is about to change internally
   //the part will then create the Map and emit completed()

   m_histories->push( m_part->url() );
}


/**********************************************
  SESSION MANAGEMENT
 **********************************************/

void
MainWindow::saveProperties( TDEConfig *config ) //virtual
{
   m_histories->save( config );
   config->writeEntry( "currentMap", m_part->url().path() );
}

void
MainWindow::readProperties( TDEConfig *config ) //virtual
{
   m_histories->restore( config );
   slotScanPath( config->readEntry( "currentMap", TQString() ) );
}

} //namespace Filelight



/// declared in historyAction.h

void setActionMenuTextOnly( TDEAction *a, TQString const &suffix )
{
    TQString const menu_text = suffix.isEmpty()
            ? a->text()
            : i18n( "&Up: /home/mxcl", "%1: %2" ).arg( a->text(), suffix );

    for (int i = 0; i < a->containerCount(); ++i) {
        TQWidget *w = a->container( i );
        int const id = a->itemId( i );

        if (w->inherits( "TQPopupMenu" ))
            static_cast<TQPopupMenu*>(w)->changeItem( id, menu_text );

        else if (w->inherits( "TDEToolBar" )) {
            TQWidget *button = static_cast<TDEToolBar*>(w)->getWidget( id );
            if (button->inherits( "TDEToolBarButton" ))
                TQToolTip::add( button, suffix );
        }
    }
}

#include "mainWindow.moc"
