/**************************************************************************
*   Copyright (C) 2006 by Michel Ludwig (michel.ludwig@kdemail.net)       *
***************************************************************************/

/**************************************************************************
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
***************************************************************************/

#include "editorkeysequencemanager.h"

#include "kileinfo.h"
#include "kilejscript.h"

#include <kate/document.h>
#include <tdelocale.h>

#define MAX(a,b) (a >= b ? a : b)

namespace KileEditorKeySequence {

	Manager::Manager(KileInfo* kileInfo, TQObject *parent, const char *name) : TQObject(parent, name), m_kileInfo(kileInfo) {
	}

	Manager::~Manager() {
	}

	void Manager::addAction(const TQString& seq, Action *action) {
		if(seq.isEmpty()) {
			return;
		}
		if(m_actionMap.find(seq) == m_actionMap.end()) {
			m_actionMap[seq] = action;
			m_watchedKeySequencesList.push_back(seq);
			emit watchedKeySequencesChanged();
		}
	}


	void Manager::removeKeySequence(const TQString& seq) {
		if(seq.isEmpty()) {
			return;
		}
		TQMap<TQString, Action*>::iterator it = m_actionMap.find(seq);
		if(it != m_actionMap.end()) {
			delete (it.data());
			m_actionMap.remove(it);
			m_watchedKeySequencesList.remove(seq);
			emit watchedKeySequencesChanged();
		}
	}

	void Manager::removeKeySequence(const TQStringList& l) {
		bool changed = false;
		for(TQStringList::const_iterator i = l.begin(); i != l.end(); ++i) {
			if((*i).isEmpty()) {
				continue;
			}
			TQMap<TQString, Action*>::iterator it = m_actionMap.find(*i);
			if(it != m_actionMap.end()) {
				delete (it.data());
				m_actionMap.remove(it);
				m_watchedKeySequencesList.remove(*i);
				changed = true;
			}
		}
		if(changed) {
			emit watchedKeySequencesChanged();
		}
	}

	void Manager::addActionMap(const TQMap<TQString, Action*>& map) {
		bool changed = false;
		for(TQMap<TQString, Action*>::const_iterator i = map.begin(); i != map.end(); ++i) {
			if(i.key().isEmpty()) {
				continue;
			}
			if(m_actionMap[i.key()] != i.data()) {
				m_actionMap[i.key()] = i.data();
				changed = true;
			}
		}
		if(changed) {
			emit watchedKeySequencesChanged();
		}
	}

	TQString Manager::getKeySequence(const Action* a) {
		for(TQMap<TQString, Action*>::const_iterator i = m_actionMap.begin(); i != m_actionMap.end(); ++i) {
			if(i.data() == a) {
				return i.key();
			}
		}
		return TQString();
	}

	Action* Manager::getAction(const TQString& seq) {
		TQMap<TQString, Action*>::iterator i = m_actionMap.find(seq);
		return (i == m_actionMap.end()) ? 0L : (*i);
	}

	void Manager::setEditorKeySequence(const TQString& /* seq */, Action* /* action */) {
	}

	void Manager::keySequenceTyped(const TQString& seq) {
		m_actionMap[seq]->execute();
	}

	void Manager::clear() {
		m_watchedKeySequencesList.clear();
		m_actionMap.clear();
		emit watchedKeySequencesChanged();
	}


	const TQStringList& Manager::getWatchedKeySequences() {
		return m_watchedKeySequencesList;
	}

	bool Manager::isSequenceAssigned(const TQString& seq) const {
		for(TQValueList<TQString>::const_iterator i = m_watchedKeySequencesList.begin(); i != m_watchedKeySequencesList.end(); ++i) {
			if((*i).startsWith(seq)) {
				return true;
			}
		}
		return false;
	}

	TQPair<int, TQString> Manager::checkSequence(const TQString& seq, const TQString& skip) {
		for(TQValueList<TQString>::iterator i = m_watchedKeySequencesList.begin(); i != m_watchedKeySequencesList.end(); ++i) {
			if((*i) == skip) {
				continue;
			}
			if((*i).startsWith(seq)) {
				return (*i == seq) ? qMakePair<int, TQString>(1, seq) : qMakePair<int, TQString>(2, *i);
			}
 			if(!(*i).isEmpty() && seq.startsWith(*i)) {
				return qMakePair<int, TQString>(3, *i);
			}
		}
		return qMakePair<int, TQString>(0, TQString());
	}

Recorder::Recorder(Kate::View *view, Manager *manager) : TQObject(view), m_manager(manager), m_view(view) {
	connect(m_manager, TQ_SIGNAL(watchedKeySequencesChanged()), this, TQ_SLOT(reloadWatchedKeySequences()));
	connect(this, TQ_SIGNAL(detectedTypedKeySequence(const TQString&)), m_manager, TQ_SLOT(keySequenceTyped(const TQString&)));
	m_view->cursorPositionReal(&m_oldLine, &m_oldCol);
	reloadWatchedKeySequences();
}

Recorder::~Recorder() {
}

bool Recorder::eventFilter(TQObject* /* o */, TQEvent *e) {
	if (e->type() == TQEvent::KeyPress) {
		TQKeyEvent *keyEvent = (TQKeyEvent*)(e);
		uint curLine, curCol;
		m_view->cursorPositionReal(&curLine, &curCol);
		if(curLine != m_oldLine || m_oldCol+1 != curCol) {
			m_typedSequence = TQString();
			m_oldLine = curLine;
			m_oldCol = curCol;
		}
		else {
			++m_oldCol;
		}
		m_typedSequence += keyEvent->text();
		if(m_typedSequence.length() == m_maxSequenceLength+1) {
			m_typedSequence = m_typedSequence.mid(1, m_typedSequence.length() - 1);
		}
		return seekForKeySequence(m_typedSequence);
	}
	return false;
}

	bool Recorder::seekForKeySequence(const TQString& s) {
		for(uint i = 0; i < s.length(); ++i) {
			TQString toCheck = s.right(s.length() - i);
			if(m_watchedKeySequencesList.contains(toCheck) > 0) {
 				m_view->getDoc()->removeText(m_oldLine, m_oldCol-(s.length() - i - 1), m_oldLine, m_oldCol);
				m_typedSequence = TQString(); // clean m_typedSequence to avoid wrong action triggering if one presses keys without printable character
				emit detectedTypedKeySequence(toCheck);
				return true;
			}
		}
		return false;
	}

	void Recorder::reloadWatchedKeySequences() {
		m_watchedKeySequencesList = m_manager->getWatchedKeySequences();
		m_maxSequenceLength = 0;
		for(TQStringList::iterator i = m_watchedKeySequencesList.begin(); i != m_watchedKeySequencesList.end(); ++i) {
			m_maxSequenceLength = MAX(m_maxSequenceLength, (*i).length());
		}
		if(m_maxSequenceLength < m_typedSequence.length()) {
			m_typedSequence = m_typedSequence.right(m_maxSequenceLength);
		}
	}

	Action::Action() {
	}

	Action::~Action() {
	}

	TQString Action::getDescription() const {
		return TQString();
	}

	ExecuteJScriptAction::ExecuteJScriptAction(KileJScript::JScript *jScript, KileJScript::Manager *jScriptManager) : m_jScript(jScript), m_jScriptManager(jScriptManager){
	}

	ExecuteJScriptAction::~ExecuteJScriptAction() {
	}

	void ExecuteJScriptAction::execute() {
		m_jScriptManager->executeJScript(m_jScript);
	}

	TQString ExecuteJScriptAction::getDescription() const {
		return i18n("Script execution of %1").arg(m_jScript->getFileName());
	}


}

#include "editorkeysequencemanager.moc"
