//=============================================================================
//
//   File : savethemedialog.cpp
//   Created on Wed 03 Jan 2007 03:01:34 by Szymon Stefanek
//
//   This file is part of the KVIrc IRC Client distribution
//   Copyright (C) 2007 Szymon Stefanek <pragma at kvirc dot net>
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
//
//=============================================================================

#include "savethemedialog.h"
#include "themefunctions.h"

#include <tqlayout.h>
#include <tqpushbutton.h>
#include <tqlineedit.h>
#include <tqregexp.h>
#include <tqmessagebox.h>
#include <tqdir.h>
#include <tqcombobox.h>
#include <tqpainter.h>
#include <tqtooltip.h>
#include <tqimage.h>
#include <kvi_tal_textedit.h>
#include <tqmultilineedit.h>
#include <tqbuffer.h>
#include <tqlabel.h>


#include "kvi_options.h"
#include "kvi_locale.h"
#include "kvi_config.h"
#include "kvi_fileutils.h"
#include "kvi_app.h"
#include "kvi_frame.h"
#include "kvi_iconmanager.h"
#include "kvi_styled_controls.h"
#include "kvi_packagefile.h"
#include "kvi_fileextensions.h"
#include "kvi_filedialog.h"
#include "kvi_msgbox.h"
#include "kvi_selectors.h"
#include "kvi_miscutils.h"
#include "kvi_sourcesdate.h"



KviSaveThemeDialog::KviSaveThemeDialog(TQWidget * pParent)
: KviTalWizard(pParent)
{
	setCaption(__tr2qs_ctx("Save Current Theme - KVIrc","theme"));
	setMinimumSize(400,350);

	// welcome page ==================================================================================
	TQWidget * pPage = new TQWidget(this);
	TQGridLayout * pLayout = new TQGridLayout(pPage,2,1,4,4);
	
	TQLabel * pLabel = new TQLabel(pPage);
	TQString szText = "<p>";
	szText += __tr2qs_ctx("This procedure allows you to save the current theme settings to a single directory. It is useful if you want to apply other themes or play with the theme settings and later come back to this theme with a single click. It will also allow you to manually modify the theme settings and later export them to a distributable package.","theme");
	szText += "</p><p>";
	szText += __tr2qs_ctx("You will be asked to provide a theme name, a description and, if you want, a screenshot.","theme");
	szText += "</p><p>";
	szText += __tr2qs_ctx("Hit the \"Next\" button to begin.","theme");
	szText += "<p>";
	
	pLabel->setText(szText);
	pLayout->addWidget(pLabel,0,0);
	pLayout->setRowStretch(1,1);
	
	addPage(pPage,__tr2qs_ctx("Welcome","theme"));
	setBackEnabled(pPage,false);
	setNextEnabled(pPage,true);
	setHelpEnabled(pPage,false);
	setFinishEnabled(pPage,false);

	// packager informations ================================================================================

	pPage = new TQWidget(this);
	pLayout = new TQGridLayout(pPage,5,2,4,4);

	pLabel = new TQLabel(pPage);
	pLabel->setText(__tr2qs_ctx("Here you need to provide informations about you (the author) and a short description of the theme you're creating.","theme"));
	pLabel->setTextFormat(TQt::RichText);
	pLayout->addMultiCellWidget(pLabel,0,0,0,1);
	
	pLabel = new TQLabel(pPage);
	pLabel->setText(__tr2qs_ctx("Theme Name:","theme"));
	pLayout->addWidget(pLabel,1,0);
	
	m_pThemeNameEdit = new TQLineEdit(pPage);
	//m_pThemeNameEdit->setText(szThemeName);
	pLayout->addWidget(m_pThemeNameEdit,1,1);

	pLabel = new TQLabel(pPage);
	pLabel->setText(__tr2qs_ctx("Version:","theme"));
	pLayout->addWidget(pLabel,2,0);
	
	m_pThemeVersionEdit = new TQLineEdit(pPage);
	//m_pThemeVersionEdit->setText(szThemeVersion);
	pLayout->addWidget(m_pThemeVersionEdit,2,1);

	pLabel = new TQLabel(pPage);
	pLabel->setText(__tr2qs_ctx("Description:","theme"));
	pLayout->addWidget(pLabel,3,0);
	
	m_pThemeDescriptionEdit = new KviTalTextEdit(pPage);
	//m_pThemeDescriptionEdit->setText(szThemeDescription);
	pLayout->addWidget(m_pThemeDescriptionEdit,3,1);

	pLabel = new TQLabel(pPage);
	pLabel->setText(__tr2qs_ctx("Theme Author:","theme"));
	pLayout->addWidget(pLabel,4,0);
	
	m_pAuthorNameEdit = new TQLineEdit(pPage);
	//m_pAuthorNameEdit->setText(szThemeAuthor);
	pLayout->addWidget(m_pAuthorNameEdit,4,1);


	pLayout->setRowStretch(3,1);
	pLayout->setColStretch(1,1);

	addPage(pPage,__tr2qs_ctx("Theme Informations","theme"));
	setBackEnabled(pPage,true);
	setHelpEnabled(pPage,false);
	setNextEnabled(pPage,true);
	setFinishEnabled(pPage,false);

	// screenshot/logo/icon ================================================================================

	pPage = new TQWidget(this);
	pLayout = new TQGridLayout(pPage,4,1,4,4);

	pLabel = new TQLabel(pPage);
	pLabel->setText(__tr2qs_ctx("Here you can either choose a screenshot image from disk or make one now. The screenshot will be displayed in the tooltips of the theme management dialog and will be also visible in the package installation dialog if you will export the theme to a distributable package.","theme"));
	pLabel->setTextFormat(TQt::RichText);
	pLayout->addWidget(pLabel,0,0);
	
	m_pImageLabel = new TQLabel(pPage);
	m_pImageLabel->setFrameStyle(TQFrame::Sunken | TQFrame::Panel);
	m_pImageLabel->setMinimumSize(300,225);
	m_pImageLabel->setAlignment(TQt::AlignCenter | TQt::AlignVCenter);
	pLayout->addWidget(m_pImageLabel,1,0);

	TQString szFilter = "*.png *.jpg *.xpm";
	m_pImageSelector = new KviFileSelector(pPage,"",&m_szScreenshotPath,true,0,szFilter);
	connect(m_pImageSelector,TQ_SIGNAL(selectionChanged(const TQString &)),this,TQ_SLOT(imageSelectionChanged(const TQString &)));
	pLayout->addWidget(m_pImageSelector,2,0);

	TQPushButton * pButton = new TQPushButton(pPage);
	pButton->setText(__tr2qs_ctx("Make Screenshot Now","theme"));
	connect(pButton,TQ_SIGNAL(clicked()),this,TQ_SLOT(makeScreenshot()));
	pLayout->addWidget(pButton,3,0);

	pLayout->setRowStretch(1,1);

	m_pImageSelectionPage = pPage;
	addPage(pPage,__tr2qs_ctx("Screenshot","theme"));
	setBackEnabled(pPage,true);
	setHelpEnabled(pPage,false);
	setNextEnabled(pPage,true);
	setFinishEnabled(pPage,true);
}

KviSaveThemeDialog::~KviSaveThemeDialog()
{
}

void KviSaveThemeDialog::imageSelectionChanged(const TQString &szImagePath)
{
	TQImage pix(szImagePath);
	if(!pix.isNull())
	{
		TQPixmap out;
		if(pix.width() > 300 || pix.height() > 225)
			out.convertFromImage(pix.smoothScale(300,225,TQImage::ScaleMin));
		else
			out.convertFromImage(pix);
		m_pImageLabel->setPixmap(out);
		return;
	}

	TQMessageBox::critical(this,__tr2qs_ctx("Save Current Theme - KVIrc","theme"),__tr2qs_ctx("Failed to load the selected image","theme"),
		TQMessageBox::Ok,TQMessageBox::NoButton,TQMessageBox::NoButton);

	m_pImageSelector->setSelection("");
	m_pImageLabel->setPixmap(TQPixmap());
}


void KviSaveThemeDialog::accept()
{
	if(!saveTheme())return;
	KviTalWizard::accept();
}


void KviSaveThemeDialog::makeScreenshot()
{
	TQString szFileName;
	g_pApp->getTmpFileName(szFileName,"screenshot.png");
	if(!KviThemeFunctions::makeKVIrcScreenshot(szFileName))
	{
		TQMessageBox::critical(this,__tr2qs_ctx("Acquire Screenshot - KVIrc","theme"),__tr2qs_ctx("Failed to make screenshot","theme"),
			TQMessageBox::Ok,TQMessageBox::NoButton,TQMessageBox::NoButton);
		return;
	}
	m_pImageSelector->setSelection(szFileName);
	imageSelectionChanged(szFileName);
}

bool KviSaveThemeDialog::saveTheme()
{
	m_pImageSelector->commit();

	KviThemeInfo sto;
	sto.setName(m_pThemeNameEdit->text());
	if(sto.name().isEmpty())
	{
		TQMessageBox::critical(this,__tr2qs_ctx("Save Current Theme - KVIrc","theme"),__tr2qs_ctx("You must choose a theme name!","theme"),TQMessageBox::Ok,
			TQMessageBox::NoButton,TQMessageBox::NoButton);
		return false;
	}

	sto.setAuthor(m_pAuthorNameEdit->text());
	sto.setDescription(m_pThemeDescriptionEdit->text());
	sto.setDate(TQDateTime::currentDateTime().toString());
	sto.setVersion(m_pThemeVersionEdit->text());
	sto.setApplication("KVIrc " KVI_VERSION "." KVI_SOURCES_DATE);

	if(sto.version().isEmpty())sto.setVersion("1.0.0");

	TQString szSubdir = sto.name() + TQString("-") + sto.version();
	szSubdir.replace(TQRegExp("[ \\\\/:][ \\\\/:]*"),"_");
	sto.setSubdirectory(szSubdir);

	TQString szAbsDir;
	g_pApp->getLocalKvircDirectory(szAbsDir,KviApp::Themes,sto.subdirectory(),true);
	if(!KviFileUtils::makeDir(szAbsDir))
	{
		TQMessageBox::critical(this,__tr2qs_ctx("Save Current Theme - KVIrc","theme"),__tr2qs_ctx("Unable to create theme directory.","theme"),
			TQMessageBox::Ok,TQMessageBox::NoButton,TQMessageBox::NoButton);
		return false;
	}

	sto.setAbsoluteDirectory(szAbsDir);

	if(!KviTheme::save(sto))
	{
		TQString szMsg2;
		TQString szErr = sto.lastError();
		KviTQString::sprintf(szMsg2,__tr2qs_ctx("Unable to save theme: %Q","theme"),&szErr);
		TQMessageBox::critical(this,__tr2qs_ctx("Save Current Theme - KVIrc","theme"),szMsg2,
			TQMessageBox::Ok,TQMessageBox::NoButton,TQMessageBox::NoButton);
		return false;
	}
	// write down the screenshot, if needed
	
	if(!m_szScreenshotPath.isEmpty())
	{
		if(!KviTheme::saveScreenshots(sto,m_szScreenshotPath))
		{
			TQMessageBox::critical(this,__tr2qs_ctx("Save Current Theme - KVIrc","theme"),__tr2qs_ctx("Failed to load the selected screenshot image: please fix it","theme"),
				TQMessageBox::Ok,TQMessageBox::NoButton,TQMessageBox::NoButton);
			setCurrentPage(m_pImageSelectionPage);
			return false;
		}
	}

	TQString szMsg = __tr2qs_ctx("Theme saved successfully to ","theme");
	szMsg += sto.absoluteDirectory();

	TQMessageBox::information(this,__tr2qs_ctx("Save Theme - KVIrc","theme"),szMsg,TQMessageBox::Ok,
		TQMessageBox::NoButton,TQMessageBox::NoButton);

	return true;
}

