/* This file is part of the KDE project
 * Copyright (C) 2002 Nadeem Hasan <nhasan@kde.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation version 2.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 */

#include "tdefile_pcx.h"

#include <kgenericfactory.h>
#include <kdebug.h>

#include <tqdatastream.h>
#include <tqfile.h>

typedef KGenericFactory<KPcxPlugin> PcxFactory;

K_EXPORT_COMPONENT_FACTORY(tdefile_pcx, PcxFactory("tdefile_pcx"))

TQDataStream &operator>>( TQDataStream &s, PALETTE &pal )
{
  for (  int i=0; i<16; ++i )
    s >> pal.p[ i ].r >> pal.p[ i ].g >> pal.p[ i ].b;

  return s;
}

TQDataStream &operator>>( TQDataStream &s, PCXHEADER &ph )
{
  s >> ph.Manufacturer;
  s >> ph.Version;
  s >> ph.Encoding;
  s >> ph.Bpp;
  s >> ph.XMin >> ph.YMin >> ph.XMax >> ph.YMax;
  s >> ph.HDpi >> ph.YDpi;
  s >> ph.Palette;
  s >> ph.Reserved;
  s >> ph.NPlanes;
  s >> ph.BytesPerLine;
  s >> ph.PaletteInfo;
  s >> ph.HScreenSize;
  s >> ph.VScreenSize;

  return s;
}

KPcxPlugin::KPcxPlugin( TQObject *parent, const char *name,
        const TQStringList &args ) : KFilePlugin( parent, name, args )
{
  kdDebug(7034) << "PCX file meta info plugin" << endl;
  KFileMimeTypeInfo* info = addMimeTypeInfo( "image/x-pcx" );

  KFileMimeTypeInfo::GroupInfo* group =
  addGroupInfo( info, "General", i18n( "General" ) );

  KFileMimeTypeInfo::ItemInfo* item;
  item = addItemInfo( group, "Dimensions", i18n( "Dimensions" ),
      TQVariant::Size );
  setHint( item, KFileMimeTypeInfo::Size );
  setUnit( item, KFileMimeTypeInfo::Pixels );
  item = addItemInfo( group, "BitDepth", i18n( "Bit Depth" ),
      TQVariant::Int );
  setUnit( item, KFileMimeTypeInfo::BitsPerPixel );
  item = addItemInfo( group, "Resolution", i18n( "Resolution" ),
      TQVariant::Size );
  setUnit( item, KFileMimeTypeInfo::DotsPerInch );
  item = addItemInfo( group, "Compression", i18n( "Compression" ),
      TQVariant::String );
}

bool KPcxPlugin::readInfo( KFileMetaInfo& info, uint )
{
  if ( info.path().isEmpty() )
    return false;

  struct PCXHEADER header;

  TQFile f( info.path() );
  if ( !f.open( IO_ReadOnly ) )
    return false;

  TQDataStream s( &f );
  s.setByteOrder(  TQDataStream::LittleEndian );

  s >> header;

  int w = (  header.XMax-header.XMin ) + 1;
  int h = (  header.YMax-header.YMin ) + 1;
  int bpp = header.Bpp*header.NPlanes;

  KFileMetaInfoGroup group = appendGroup( info, "General" );

  appendItem( group, "Dimensions", TQSize( w, h ) );
  appendItem( group, "BitDepth", bpp );
  appendItem( group, "Resolution", TQSize( header.HDpi, header.YDpi ) );
  if ( header.Encoding == 1 )
    appendItem( group, "Compression", i18n( "Yes (RLE)" ) );
  else
    appendItem( group, "Compression", i18n( "None" ) );

  f.close();

  return true;
}

#include "tdefile_pcx.moc"
