/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 *
 * Date        : 2007-12-08
 * Description : a widget to display date and time statistics of pictures
 *
 * Copyright (C) 2007-2008 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */

#ifndef TIMELINEWIDGET_H
#define TIMELINEWIDGET_H

// TQt includes.

#include <tqstring.h>
#include <tqwidget.h>
#include <tqdatetime.h>

// Local includes.

#include "ddebug.h"

namespace Digikam
{

typedef TQPair<TQDateTime, TQDateTime> DateRange;    // Range of a contigue dates selection <start date, end date>.
typedef TQValueList<DateRange> DateRangeList;      // List of dates range selected.

class TimeLineWidgetPriv;

class TimeLineWidget : public TQWidget
{
Q_OBJECT

public:

    enum TimeUnit 
    {
        Day = 0,
        Week,
        Month,
        Year
    };

    enum SelectionMode
    {
        Unselected=0,      // No selection.
        FuzzySelection,    // Partially selected.
        Selected           // Fully selected.
    };

    enum ScaleMode
    {
        LinScale=0,        // Linear scale.
        LogScale           // Logarithmic scale.
    };

public:

    TimeLineWidget(TQWidget *parent=0);
    ~TimeLineWidget();

    void      setTimeUnit(TimeUnit timeUnit);
    TimeUnit  timeUnit() const;

    void      setScaleMode(ScaleMode scaleMode);
    ScaleMode scaleMode() const;

    void      setCursorDateTime(const TQDateTime& dateTime);
    TQDateTime cursorDateTime() const;
    int       cursorInfo(TQString& infoDate);

    /** Return a list of Date-Range based on selection performed on days-map */
    DateRangeList selectedDateRange(int& totalCount);
    void          setSelectedDateRange(const DateRangeList& list);

    int  totalIndex();
    int  indexForRefDateTime();
    int  indexForCursorDateTime();
    void setCurrentIndex(int index);

signals:

    void signalCursorPositionChanged();
    void signalSelectionChanged();
    void signalRefDateTimeChanged();
    void signalDateMapChanged();

public slots:

    void slotDatesMap(const TQMap<TQDateTime, int>&);
    void slotPrevious();
    void slotNext();
    void slotBackward();
    void slotForward();
    void slotResetSelection();

private slots:

    void slotThemeChanged();

private:

    TQDateTime     prevDateTime(const TQDateTime& dt);
    TQDateTime     nextDateTime(const TQDateTime& dt);

    int           maxCount();
    int           indexForDateTime(const TQDateTime& date);
    int           statForDateTime(const TQDateTime& dt, SelectionMode& selected);
    void          setRefDateTime(const TQDateTime& dateTime);

    void          updatePixmap();
    void          paintEvent(TQPaintEvent*);
    void          resizeEvent(TQResizeEvent*);
    void          wheelEvent(TQWheelEvent*); 

    void          mousePressEvent(TQMouseEvent*);
    void          mouseMoveEvent(TQMouseEvent*);
    void          mouseReleaseEvent(TQMouseEvent*);

    TQDateTime     dateTimeForPoint(const TQPoint& pt, bool &isOnSelectionArea);
    TQDateTime     firstDayOfWeek(int year, int weekNumber);

    void          resetSelection();
    void          setDateTimeSelected(const TQDateTime& dt, SelectionMode selected);
    void          setDaysRangeSelection(const TQDateTime dts, const TQDateTime dte, SelectionMode selected);
    SelectionMode checkSelectionForDaysRange(const TQDateTime dts, const TQDateTime dte);
    void          updateWeekSelection(const TQDateTime dts, const TQDateTime dte);
    void          updateMonthSelection(const TQDateTime dts, const TQDateTime dte);
    void          updateYearSelection(const TQDateTime dts, const TQDateTime dte);
    void          updateAllSelection();

private:

    TimeLineWidgetPriv* d;
};

}  // NameSpace Digikam

#endif // TIMELINEWIDGET_H
